package cn.wumoe.hime.module.http.server

import cn.wumoe.hime.api.scripting.HimeContext
import cn.wumoe.hime.inter.Function
import cn.wumoe.hime.inter.Module
import cn.wumoe.hime.lexer.Num
import cn.wumoe.hime.lexer.Token
import cn.wumoe.hime.lexer.Word
import cn.wumoe.hime.toWord
import com.sun.net.httpserver.HttpServer
import java.net.InetSocketAddress
import java.util.*
import java.util.concurrent.Executors

class HttpServerModule : Module("hime.http.server") {
    override fun init(context: HimeContext) {
        addFunction(HttpServerCreate())         // http-server-create
        addFunction(HttpServerStart())          // http-server-start
        addFunction(HttpServerCreateContext())  // http-server-context-create
        addFunction(HttpServerRemoveContext())  // http-server-context-remove
        addFunction(HttpServerStop())           // http-server-stop
        addFunction(HttpExchangeGetValue())     // http-server-get-value
        addFunction(HttpExchangePostValue())    // http-server-post-value
        addFunction(HttpExchangeReturnString()) // http-server-return-string
        addFunction(HttpExchangeReturnBase64()) // http-server-return-base64
        addFunction(HttpExchangeIp())           // http-server-ip
        addFunction(HttpExchangeRequestMethod())// http-server-request-method
    }

    class HttpServerCreateContext : Function("http-server-context-create") {
        override fun call(pars: Array<out Token>): Token {
            if (pars.size >= 3 && pars[0] is HimeHttpServerToken)
                (pars[0] as HimeHttpServerToken).httpserver.createContext(
                    pars[1].toString(),
                    HimeHttpServerHandler(
                        HimeHttpServerHandler.HimeHttpServerHandlerFunction(
                            analysis,
                            pars[1].toString(),
                            analysis.getFunction(pars[2].toString())
                        )
                    )
                )
            return Word.NIL
        }
    }

    class HttpServerRemoveContext : Function("http-server-context-remove") {
        override fun call(pars: Array<out Token>): Token {
            if (pars.size >= 2 && pars[0] is HimeHttpServerToken)
                (pars[0] as HimeHttpServerToken).httpserver.removeContext(pars[1].toString())
            return Word.NIL
        }
    }

    class HttpServerCreate : Function("http-server-create") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty() && pars[0] is Num) {
                val httpServer = HttpServer.create(InetSocketAddress((pars[0] as Num).value.toInt()), 0)
                httpServer.executor = Executors.newCachedThreadPool()
                HimeHttpServerToken(httpServer)
            } else
                Word.NIL
        }
    }

    class HttpServerStop : Function("http-server-stop") {
        override fun call(pars: Array<out Token>): Token {
            if (pars.size >= 2 && pars[0] is HimeHttpServerToken && pars[1] is Num)
                (pars[0] as HimeHttpServerToken).httpserver.stop((pars[1] as Num).value.toInt())
            return Word.NIL
        }
    }

    class HttpServerStart : Function("http-server-start") {
        override fun call(pars: Array<out Token>): Token {
            if (pars.isNotEmpty() && pars[0] is HimeHttpServerToken)
                (pars[0] as HimeHttpServerToken).httpserver.start()
            return Word.NIL
        }
    }

    class HttpExchangeGetValue : Function("http-server-get-value") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2 && pars[0] is HimeHttpExchangeToken) {
                val exchange = pars[0] as HimeHttpExchangeToken
                exchange.getValue[pars[1].toString()]!!.toWord()
            } else
                Word.NIL
        }
    }

    class HttpExchangeIp: Function("http-server-ip") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty() && pars[0] is HimeHttpExchangeToken) {
                val exchange = pars[0] as HimeHttpExchangeToken
                exchange.exchange.remoteAddress.address.toString().toWord()
            } else
                Word.NIL
        }
    }

    class HttpExchangeRequestMethod: Function("http-server-request-method") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty() && pars[0] is HimeHttpExchangeToken) {
                val exchange = pars[0] as HimeHttpExchangeToken
                exchange.exchange.requestMethod.toWord()
            } else
                Word.NIL
        }
    }

    class HttpExchangeReturnString : Function("http-server-return-string") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 3 && pars[0] is HimeHttpExchangeToken && pars[1] is Num) {
                val exchange = pars[0] as HimeHttpExchangeToken
                HttpHandlerUtils.result(exchange.exchange, (pars[1] as Num).value.toInt(), pars[2].toString())
                pars[2]
            } else
                Word.NIL
        }
    }

    class HttpExchangeReturnBase64 : Function("http-server-return-base64") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 3 && pars[0] is HimeHttpExchangeToken && pars[1] is Num) {
                val exchange = pars[0] as HimeHttpExchangeToken
                HttpHandlerUtils.result(exchange.exchange, (pars[1] as Num).value.toInt(), Base64.getDecoder().decode(pars[2].toString()))
                pars[2]
            } else
                Word.NIL
        }
    }

    class HttpExchangePostValue : Function("http-server-post-value") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2 && pars[0] is HimeHttpExchangeToken) {
                val exchange = pars[0] as HimeHttpExchangeToken
                exchange.postValue[pars[1].toString()]!!.toWord()
            } else
                Word.NIL
        }
    }
}