package cn.wumoe.hime.module

import com.google.zxing.*
import com.google.zxing.client.j2se.BufferedImageLuminanceSource
import com.google.zxing.client.j2se.MatrixToImageWriter
import com.google.zxing.common.HybridBinarizer
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel
import cn.wumoe.hime.api.scripting.HimeContext
import cn.wumoe.hime.inter.Function
import cn.wumoe.hime.inter.Module
import cn.wumoe.hime.lexer.Num
import cn.wumoe.hime.lexer.Token
import cn.wumoe.hime.lexer.Word
import cn.wumoe.hime.toWord
import java.awt.BasicStroke
import java.awt.geom.RoundRectangle2D
import java.io.ByteArrayInputStream
import java.io.ByteArrayOutputStream
import java.nio.charset.StandardCharsets
import java.util.*
import javax.imageio.ImageIO

class QRCodeModule :  Module("hime.qrcode") {
    override fun init(context: HimeContext) {
        addFunction(QRCodeEncodeLogo())   // qrcode-encode-logo
        addFunction(QRCodeEncode())       // qrcode-encode
        addFunction(QRCodeDecode())       // qrcode-decode
    }

    // (qrcode-decode qrcode)
    class QRCodeDecode : Function("qrcode-decode") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                val formatReader = MultiFormatReader()
                val image = ImageIO.read(ByteArrayInputStream(Base64.getDecoder().decode(pars[0].toString())))
                val binaryBitmap = BinaryBitmap(HybridBinarizer(BufferedImageLuminanceSource(image)))
                val hints = HashMap<DecodeHintType, Any>()
                hints[DecodeHintType.CHARACTER_SET] = StandardCharsets.UTF_8.displayName()
                val result = formatReader.decode(binaryBitmap, hints)
                result.text.toWord()
            } else
                Word.NIL
        }
    }

    // (qrcode-encode-logo qrcodeSize logoSize logo content)
    class QRCodeEncodeLogo : Function("qrcode-encode-logo") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 4 && pars[0] is Num) {
                val qrcodeSize = (pars[0] as Num).value.toInt()
                val logoSize = (pars[1] as Num).value.toInt()
                val content = pars[3].toString()
                val hints = HashMap<EncodeHintType, Any>()
                hints[EncodeHintType.ERROR_CORRECTION] = ErrorCorrectionLevel.H
                hints[EncodeHintType.CHARACTER_SET] = StandardCharsets.UTF_8.displayName()
                hints[EncodeHintType.MARGIN] = 1
                val bitMatrix = MultiFormatWriter().encode(
                    content, BarcodeFormat.QR_CODE, qrcodeSize, qrcodeSize,
                    hints
                )
                val image = MatrixToImageWriter.toBufferedImage(bitMatrix)
                val logoImage = ImageIO.read(ByteArrayInputStream(Base64.getDecoder().decode(pars[2].toString())))
                val graphics2D = image.createGraphics()
                val x: Int = (qrcodeSize - logoSize) / 2
                val y: Int = (qrcodeSize - logoSize) / 2
                graphics2D.drawImage(logoImage, x, y, logoSize, logoSize, null)
                val shape = RoundRectangle2D.Float(x.toFloat(), y.toFloat(), logoSize.toFloat(), logoSize.toFloat(), 6f, 6f)
                graphics2D.stroke = BasicStroke(3f)
                graphics2D.draw(shape)
                graphics2D.dispose()
                val byteArrayOutputStream = ByteArrayOutputStream()
                ImageIO.write(image, "png", byteArrayOutputStream)
                Base64.getEncoder().encodeToString(byteArrayOutputStream.toByteArray()).toWord()
            } else
                Word.NIL
        }
    }

    // (qrcode-encode qrcodeSize content)
    class QRCodeEncode : Function("qrcode-encode") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2 && pars[0] is Num) {
                val size = (pars[0] as Num).value.toInt()
                val content = pars[1].toString()
                val hints = HashMap<EncodeHintType, Any>()
                hints[EncodeHintType.ERROR_CORRECTION] = ErrorCorrectionLevel.H
                hints[EncodeHintType.CHARACTER_SET] = StandardCharsets.UTF_8.displayName()
                hints[EncodeHintType.MARGIN] = 1
                val bitMatrix = MultiFormatWriter().encode(
                    content, BarcodeFormat.QR_CODE, size, size,
                    hints
                )
                val image = MatrixToImageWriter.toBufferedImage(bitMatrix)
                val byteArrayOutputStream = ByteArrayOutputStream()
                ImageIO.write(image, "png", byteArrayOutputStream)
                Base64.getEncoder().encodeToString(byteArrayOutputStream.toByteArray()).toWord()
            } else
                Word.NIL
        }
    }
}