package cn.wumoe.hime.module

import com.google.common.math.BigIntegerMath
import cn.wumoe.hime.api.scripting.HimeContext
import cn.wumoe.hime.inter.Function
import cn.wumoe.hime.inter.Module
import cn.wumoe.hime.lexer.Num
import cn.wumoe.hime.lexer.Real
import cn.wumoe.hime.lexer.Token
import cn.wumoe.hime.lexer.Word
import cn.wumoe.hime.toNum
import cn.wumoe.hime.toReal
import java.math.BigDecimal
import java.math.BigInteger
import java.math.MathContext
import java.math.RoundingMode
import kotlin.math.cos
import kotlin.math.sin

class MathModule : Module("hime.math") {
    override fun init(context: HimeContext) {
        addFunction(Max())      // max
        addFunction(Min())      // min
        addFunction(Sqrt())     // sqrt
        addFunction(Log10())    // log10
        addFunction(Log2())     // log2
        addFunction(Abs())      // abs
        addFunction(Gcd())      // gcd
        addFunction(Lcm())      // lcm
        addFunction(Mod())      // mod
        addFunction(Pow())      // pow
        addFunction(Average())  // average
        addFunction(Ceil())     // ceil
        addFunction(Floor())    // floor
        addFunction(Sin())      // sin
        addFunction(Cos())      // cos
    }

    class Ceil : Function("ceil") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                BigInteger(BigDecimal(pars[0].toString()).setScale(0, RoundingMode.CEILING).toPlainString()).toNum()
            } else
                Word.NIL
        }
    }

    class Sin : Function("sin") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                sin(BigDecimal(pars[0].toString()).toDouble()).toReal()
            } else
                Word.NIL
        }
    }

    class Cos : Function("cos") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                cos(BigDecimal(pars[0].toString()).toDouble()).toReal()
            } else
                Word.NIL
        }
    }

    class Floor : Function("floor") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                BigInteger(BigDecimal(pars[0].toString()).setScale(0, RoundingMode.FLOOR).toPlainString()).toNum()
            } else
                Word.NIL
        }
    }

    class Average : Function("average") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                var num = BigDecimal.ZERO
                for (t in pars)
                    num = num.add(if (t is Num) BigDecimal(t.value.toString()) else (t as Real).value)
                num.divide(pars.size.toBigDecimal()).toReal()
            } else
                Word.NIL
        }
    }

    class Pow : Function("pow") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2) {
                if (pars[0] is Num)
                    (pars[0] as Num).value.pow((pars[1] as Num).value.toInt()).toNum()
                else if (pars[0] is Real)
                    (pars[0] as Real).value.pow((pars[1] as Num).value.toInt()).toReal()
                else
                    Word.NIL
            } else
                Word.NIL
        }
    }

    class Mod : Function("mod") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2 && pars[0] is Num && pars[1] is Num) {
                (pars[0] as Num).value.mod((pars[1] as Num).value).toNum()
            } else
                Word.NIL
        }
    }

    class Max : Function("max") {
        override fun call(pars: Array<out Token>): Token {
            var index = 0
            if (pars.isNotEmpty()) {
                var max = if (pars[0] is Num)
                    BigDecimal((pars[0] as Num).value)
                else if (pars[0] is Real)
                    (pars[0] as Real).value
                else
                    return Word.NIL
                for (i in 1 until pars.size) {
                    val temp = if (pars[i] is Num)
                        BigDecimal((pars[i] as Num).value)
                    else if (pars[i] is Real)
                        (pars[i] as Real).value
                    else
                        return Word.NIL
                    if (temp > max) {
                        max = temp
                        index = i
                    }
                }
            }
            return pars[index]
        }
    }

    class Min : Function("min") {
        override fun call(pars: Array<out Token>): Token {
            var index = 0
            if (pars.isNotEmpty()) {
                var min = if (pars[0] is Num)
                    BigDecimal((pars[0] as Num).value)
                else if (pars[0] is Real)
                    (pars[0] as Real).value
                else
                    return Word.NIL
                for (i in 1 until pars.size) {
                    val temp = if (pars[i] is Num)
                        BigDecimal((pars[i] as Num).value)
                    else if (pars[i] is Real)
                        (pars[i] as Real).value
                    else
                        return Word.NIL
                    if (temp < min) {
                        min = temp
                        index = i
                    }
                }
            }
            return pars[index]
        }
    }

    class Sqrt : Function("sqrt") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                if (pars[0] is Num)
                    (pars[0] as Num).value.sqrt().toNum()
                else if (pars[0] is Real)
                    (pars[0] as Real).value.sqrt(MathContext(10)).toReal()
                else
                    Word.NIL
            } else
                Word.NIL
        }
    }

    class Abs : Function("abs") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                if (pars[0] is Num)
                    (pars[0] as Num).value.abs().toNum()
                else if (pars[0] is Real)
                    (pars[0] as Real).value.abs().toReal()
                else
                    Word.NIL
            } else
                Word.NIL
        }
    }

    class Gcd : Function("gcd") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2 && pars[0] is Num && pars[1] is Num) {
                var temp = (pars[0] as Num).value.gcd((pars[1] as Num).value)
                for (i in 2 until pars.size)
                    if (pars[i] !is Num)
                        return Word.NIL
                    else
                        temp = temp.gcd((pars[i] as Num).value)

                temp.toNum()
            } else
                Word.NIL
        }
    }

    class Lcm : Function("lcm") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2 && pars[0] is Num && pars[1] is Num) {
                var temp = (pars[0] as Num).value.lcm((pars[1] as Num).value)
                for (i in 2 until pars.size)
                    if (pars[i] !is Num)
                        return Word.NIL
                    else
                        temp = temp.lcm((pars[i] as Num).value)

                temp.toNum()
            } else
                Word.NIL
        }

        private fun BigInteger.lcm(n: BigInteger): BigInteger =
            (this.multiply(n).abs()).divide(this.gcd(n))
    }

    class Log10 : Function("log10") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty() && pars[0] is Num) {
                Num.toNum(BigIntegerMath.log10((pars[0] as Num).value, RoundingMode.CEILING))
            } else
                Word.NIL
        }
    }

    class Log2 : Function("log2") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty() && pars[0] is Num) {
                Num.toNum(BigIntegerMath.log2((pars[0] as Num).value, RoundingMode.CEILING))
            } else
                Word.NIL
        }
    }
}