package cn.wumoe.hime.semantic

import cn.wumoe.hime.*
import cn.wumoe.hime.function.CustomFun
import cn.wumoe.hime.gui.HimeTokenMaker
import cn.wumoe.hime.inter.ASTNode
import cn.wumoe.hime.inter.Function
import cn.wumoe.hime.inter.Module
import cn.wumoe.hime.lexer.*
import cn.wumoe.hime.lexer.Array
import cn.wumoe.hime.parser.Parser
import java.math.BigDecimal
import java.math.BigInteger
import javax.script.Bindings

typealias HimeEntry = MutableMap.MutableEntry<String, Any?>

class SymbolList(
    var moduleMap: MutableMap<String, Module>,
    var variableMap: MutableMap<String, Token>,
    var astNode: ASTNode, var father: SymbolList?,
    var analysis: Analysis? = null
) : Token(Tag.SYMBOL_LIST), Bindings {
    fun removeModule(module: Module) {
        var data = this
        while (data.father != null && data !== Analysis.overallData && !data.moduleMap.containsKey(module.path)) {
            data = data.father!!
        }
        data.moduleMap.remove(module.path)
    }

    fun removeFunction(function: Function) {
        var data = this
        while (data.father != null && data !== Analysis.overallData && !data.variableMap.containsKey(function.name))
            data = data.father!!
        data.variableMap.remove(function.name)
    }

    fun removeVariable(key: String) {
        var data = this
        while (data.father != null && data !== Analysis.overallData && !data.variableMap.containsKey(key)) {
            data = data.father!!
        }
        data.variableMap.remove(key)
    }

    fun setVariable(key: String, value: Token) {
        var data = this
        while (data.father != null && data !== Analysis.overallData && !data.variableMap.containsKey(key)) {
            data = data.father!!
        }
        data.variableMap[key] = value
    }

    fun setFunction(func: Function) {
        var data = this
        while (data.father != null && data !== Analysis.overallData && !data.variableMap.containsKey(func.name))
            data = data.father!!
        if (data.variableMap[func.name] is Function)
            data.variableMap[func.name] = func
    }

    fun setModule(module: Module) {
        var data = this
        while (data.father != null && data !== Analysis.overallData && !data.moduleMap.containsKey(module.path)) {
            data = data.father!!
        }
        data.moduleMap[module.path] = module
    }

    fun addFunction(func: Function) {
        if (Config.GUI)
            HimeTokenMaker.tokenMap.put(func.name, org.fife.ui.rsyntaxtextarea.Token.FUNCTION)
        variableMap[func.name] = func
    }

    fun addModule(module: Module) {
        moduleMap[module.path] = module
    }

    fun addVariable(key: String, value: Token) {
        if (Config.GUI)
            HimeTokenMaker.tokenMap.put(key, org.fife.ui.rsyntaxtextarea.Token.VARIABLE)
        variableMap[key] = value
    }

    fun containsVariable(s: String): Boolean {
        var data = this
        while (data.father != null && data !== Analysis.overallData && !data.variableMap.containsKey(s))
            data = data.father!!
        return data.variableMap.containsKey(s)
    }

    fun containsFunction(s: String): Boolean {
        var data = this
        while (data.father != null && data !== Analysis.overallData && !data.variableMap.containsKey(s))
            data = data.father!!
        return data.variableMap.containsKey(s) && data.variableMap[s] is Function
    }

    fun containsModule(s: String): Boolean {
        var data = this
        while (data.father != null && data !== Analysis.overallData && !data.moduleMap.containsKey(s)) {
            data = data.father!!
        }
        return data.moduleMap.containsKey(s)
    }

    fun getModule(s: String): Module? {
        var data = this
        while (data.father != null && data !== Analysis.overallData && !data.moduleMap.containsKey(s)) {
            data = data.father!!
        }
        return data.moduleMap[s]
    }

    fun getVariable(s: String): Token? {
        var data = this
        while (data.father != null && data !== Analysis.overallData && !data.variableMap.containsKey(s)) {
            data = data.father!!
        }
        return data.variableMap[s]
    }

    fun getFunction(s: String): Function? {
        var data = this
        while (data.father != null && data !== Analysis.overallData && !data.variableMap.containsKey(s))
            data = data.father!!
        return if (data.variableMap[s] is Function)
            data.variableMap[s] as Function
        else
            null

    }

    private fun provideFunction(k: String, v: String) {
        val s = Format.format(v)
        val lex = Lexer()
        lex.pushData(s)
        val parser = Parser(lex)
        val asts = parser.program().toTypedArray()
        CustomFun(analysis!!, k, asts, arrayOf())
    }

    override fun clear() {
        variableMap.clear()
    }

    override fun put(key: String, value: Any?): Any? {
        if (value != null) {
            if (!value.toString().trim().startsWith("("))
                addVariable(key, value.toToken())
            else {
                var size = 0
                for (c in value.toString().trim().toCharArray())
                    if (c == '(') ++size else if (c == ')') --size
                if (size != 0)
                    addVariable(key, value.toToken())
                else
                    provideFunction(key, value.toString())
            }
        }
        return value
    }

    override fun putAll(from: Map<out String, Any>) = from.forEach { (k, v) ->put(k, v) }

    override fun remove(key: String?): Any? = variableMap.remove(key)

    override fun containsKey(key: String?): Boolean = variableMap.containsKey(key)

    override fun containsValue(value: Any?): Boolean {
        for ((_, u) in variableMap)
            if (u == value)
                return true
        return false
    }

    override fun get(key: String?): Any = cast<Token>(variableMap[key])
    override fun isEmpty(): Boolean = variableMap.isEmpty()
    
    override val entries: MutableSet<HimeEntry>
        get() = object : AbstractMutableSet<HimeEntry>() {
            override val size: Int get() = variableMap.size
            override fun add(element: HimeEntry): Boolean {
                variableMap[element.key] = element.value!!.toToken()
                return true
            }

            override operator fun iterator(): MutableIterator<HimeEntry> =
                object : MutableIterator<HimeEntry> {
                    val it = variableMap.iterator()
                    override fun hasNext(): Boolean = it.hasNext()
                    override fun remove() = it.remove()
                    override fun next(): HimeEntry =
                        it.next().let { e ->
                            object : HimeEntry {
                                override val value: Any get() = e.value
                                override val key: String get() = e.key
                                override fun setValue(newValue: Any?) =
                                    e.setValue(e.value)
                            }
                        }
                }

        }
    override val keys: MutableSet<String>
        get() = variableMap.keys
    override val values: MutableList<Any?>
        get() = mutableListOf<Any?>().apply { variableMap.values.forEach { forceRun { add(it) } } }
    override val size: Int
        get() = variableMap.size
}