package cn.wumoe.hime.module

import cn.wumoe.hime.*
import cn.wumoe.hime.api.scripting.HimeContext
import cn.wumoe.hime.exceptions.HimeRuntimeException
import cn.wumoe.hime.gui.GUI
import cn.wumoe.hime.gui.HimeTokenMaker
import cn.wumoe.hime.inter.ASTNode
import cn.wumoe.hime.inter.Function
import cn.wumoe.hime.inter.Module
import cn.wumoe.hime.lexer.*
import cn.wumoe.hime.parser.Parser
import cn.wumoe.hime.semantic.Analysis
import cn.wumoe.hime.function.CustomFun
import cn.wumoe.hime.semantic.SymbolList
import java.math.BigDecimal
import java.math.BigInteger
import java.math.RoundingMode
import java.util.*
import java.util.concurrent.ThreadLocalRandom
import kotlin.Array

class CoreModule(context: HimeContext) : Module("hime.core", context) {
    override fun init(context: HimeContext) {
        addFunction(Println(context))// println
        addFunction(Print(context)) // print
        addFunction(CoreSum())      // +
        addFunction(CoreSub())      // -
        addFunction(CoreDiv())      // /
        addFunction(CoreMult())     // *
        addFunction(CoreList())     // list
        addFunction(CoreAdd())      // add
        addFunction(CoreGet())      // get
        addFunction(CoreSet())      // set
        addFunction(CoreRange())    // range
        addFunction(CoreEq())       // =
        addFunction(CoreNotEq())    // !=
        addFunction(CoreGreater())  // >
        addFunction(CoreLess())     // <
        addFunction(CoreGreaterEq())// >=
        addFunction(CoreLessEq())   // <=
        addFunction(CoreAnd())      // and
        addFunction(CoreOr())       // or
        addFunction(CoreNot())      // not
        addFunction(CoreCons())     // cons
        addFunction(CoreCar())      // car
        addFunction(CoreCdr())      // cdr
        addFunction(CoreAppend())   // append
        addFunction(CoreConcat())   // concat
        addFunction(CoreLen())      // len
        addFunction(CoreReadLine(context)) // read-line
        addFunction(CoreRead(context))     // read
        addFunction(CoreReadNum(context))  // read-num
        addFunction(CoreReadReal(context)) // read-real
        addFunction(CoreReadBool(context)) // read-bool
        addFunction(CoreToString()) // to-string
        addFunction(CoreToNum())    // to-num
        addFunction(CoreToReal())   // to-real
        addFunction(CoreLambda())   // lambda
        addFunction(CoreBegin())    // begin
        addFunction(CoreIf())       // if
        addFunction(CoreCond())     // cond
        addFunction(CoreDef())      // def
        addFunction(CoreModule())   // module
        addFunction(CoreNewLine())  // newline
        addFunction(CoreRandom())   // random
        addFunction(CoreError())    // error
        addFunction(CoreReturn())   // return
        addFunction(CoreEval())     // eval
    }

    class CoreEval : Function("eval") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                val temp = analysis.copy()
                temp.eval = true
                val value = pars[0].toString()
                val lex = Lexer()
                lex.pushData(value)
                val parser = Parser(lex)
                val asts = parser.program()
                for (node in asts)
                    temp.call(node)
                asts[asts.size - 1].tok
            } else
                Word.NIL
        }
    }

    class CoreReturn : Function("return") {
        override fun call(pars: Array<out Token>): Token {
            analysis.end = true
            return Word.NIL
        }
    }

    class CoreError : Function("error") {
        override fun call(pars: Array<out Token>): Token {
            throw HimeRuntimeException(pars[0].toString())
        }
    }

    class CoreRandom : Function("random") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2 && pars[0] is Num && pars[1] is Num)
                randomBigInteger((pars[0] as Num).value, (pars[1] as Num).value).toNum()
            else
                Word.NIL
        }

        private fun randomBigInteger(start: BigInteger, end: BigInteger): BigInteger {
            val rand = Random()
            val scale = end.toString().length
            var generated = ""
            for (i in 0 until end.toString().length)
                generated += rand.nextInt(10)
            val inputRangeStart = BigDecimal("0").setScale(scale, RoundingMode.FLOOR)
            val inputRangeEnd =
                BigDecimal(String.format("%0" + end.toString().length + "d", 0).replace('0', '9')).setScale(
                    scale,
                    RoundingMode.FLOOR
                )
            val outputRangeStart = BigDecimal(start).setScale(scale, RoundingMode.FLOOR)
            val outputRangeEnd = BigDecimal(end).add(BigDecimal("1"))
                .setScale(scale, RoundingMode.FLOOR)
            val bd1 = BigDecimal(BigInteger(generated)).setScale(scale, RoundingMode.FLOOR).subtract(inputRangeStart)
            val bd2 = inputRangeEnd.subtract(inputRangeStart)
            val bd3 = bd1.divide(bd2, RoundingMode.FLOOR)
            val bd4 = outputRangeEnd.subtract(outputRangeStart)
            val bd5 = bd3.multiply(bd4)
            val bd6 = bd5.add(outputRangeStart)
            var returnInteger = bd6.setScale(0, RoundingMode.FLOOR).toBigInteger()
            returnInteger =
                if (returnInteger > end) end else returnInteger
            return returnInteger
        }
    }

    class CoreNewLine : Function("newline") {
        override fun call(pars: Array<out Token>): Token {
            println()
            return Word.NIL
        }
    }

    class CoreModule : Function("module", true) {
        override fun call(pars: Array<out Token>): Token {
            val ast = pars[0] as ASTNode
            val temp: Analysis = analysis.copy()
            val module: Module? =
                if ((pars[1] as SymbolList).containsModule(name)) (pars[1] as SymbolList).getModule(name) else object :
                    Module(name) {
                    override fun init(context: HimeContext) {}
                }
            Analysis.module = module
            for (astNode in ast.prev)
                temp.call(astNode)
            return ast.tok
        }
    }

    class CoreDef : Function("def", true) {
        override fun call(pars: Array<out Token>): Token {
            val ast = pars[0] as ASTNode
            if (ast.size() >= 2) {
                val temp: Analysis = analysis.copy()
                if (ast[0].tok is Word) {
                    if (ast[0].size() >= 1 || ast[0].tag == ASTNode.AstTag.FUNCTION) {
                        val prs = arrayOfNulls<String>(ast[0].size())
                        for (i in 0 until ast[0].size())
                            prs[i] = ast[0][i].tok.toString()
                        val asts = arrayOfNulls<ASTNode>(ast.size() - 1)
                        for (i in 1 until ast.size())
                            asts[i - 1] = ast[i]
                        if (Analysis.module == null)
                            (pars[1] as SymbolList).addFunction(
                                CustomFun(
                                    temp,
                                    ast[0].tok.toString(),
                                    asts,
                                    prs
                                )
                            ) else
                                Analysis.module.addFunction(CustomFun(temp, ast[0].tok.toString(), asts, prs))
                    } else {
                        temp.call(ast[1])
                        if (Config.GUI) {
                            HimeTokenMaker.tokenMap.put(
                                ast[0].tok.toString(),
                                org.fife.ui.rsyntaxtextarea.Token.VARIABLE
                            )
                        }
                        if (Analysis.module == null)
                                (pars[1] as SymbolList).addVariable(
                            ast[0].tok.toString(),
                            ast[1].tok
                        ) else
                            Analysis.module.addVariable(ast[0].tok.toString(), ast[1])
                    }
                }
            }
            ast.tok = Word.NIL
            ast.clear()
            return ast.tok
        }
    }

    class CoreIf : Function("if", true) {
        override fun call(pars: Array<out Token>): Token {
            val ast = pars[0] as ASTNode
            if (ast.size() >= 1) {
                val temp: Analysis = analysis.copy()
                temp.call(ast[0])
                if (ast[0].tok == Word.True) {
                    temp.call(ast[1])
                    ast.tok = ast[1].tok
                    ast.clear()
                } else {
                    if (ast.size() < 3) {
                        ast.tok = Word.NIL
                        ast.clear()
                    } else {
                        temp.call(ast[2])
                        ast.tok = ast[2].tok
                        ast.clear()
                    }
                }
            } else {
                ast.tok = Word.NIL
                ast.clear()
            }
            return ast.tok
        }
    }

    class CoreCond : Function("cond", true) {
        override fun call(pars: Array<out Token>): Token {
            val ast = pars[0] as ASTNode
            val temp = analysis.copy()
            var ok = false
            for (astNode in ast.prev) {
                if (astNode.tok === Word.ELSE) {
                    temp.call(astNode[0])
                    ast.tok = astNode[0].tok
                    ast.clear()
                    ok = true
                } else {
                    temp.call(astNode[0])
                    if (astNode[0].tok === Word.True || astNode.tok === Word.ELSE) {
                        temp.call(astNode[1])
                        ast.tok = astNode[1].tok
                        ast.clear()
                        ok = true
                        break
                    }
                }
            }
            if (!ok) {
                ast.tok = Word.NIL
                ast.clear()
            }
            return ast.tok
        }
    }

    class CoreBegin : Function("begin", true) {
        override fun call(pars: Array<out Token>): Token {
            val ast = pars[0] as ASTNode
            val temp: Analysis = analysis.copy()
            for (astNode in ast.prev)
                temp.call(astNode)
            ast.tok = ast.prev[ast.prev.size - 1].tok
            ast.clear()
            return ast.tok
        }
    }

    class CoreLambda : Function("lambda", true) {
        override fun call(pars: Array<out Token>): Token {
            val ast = pars[0] as ASTNode
            val temp: Analysis = analysis.copy()
            val name = getRandomFunctionName()
            if (ast.size() >= 2 && ast[0] != ASTNode.EMPTY) {
                val prs = arrayOfNulls<String>(ast[0].size() + 1)
                prs[0] = ast[0].tok.toString()
                for (i in 0 until ast[0].size())
                    prs[i + 1] = ast[0][i].tok.toString()
                val asts = arrayOfNulls<ASTNode>(ast.size() - 1)
                for (i in 1 until ast.size())
                    asts[i - 1] = ast[i]
                Analysis.overallData.addFunction(CustomFun(temp, name, asts, prs))
                ast.tok = Word(name, Tag.ID)
                ast.clear()
            } else if (ast.size() >= 2) {
                val asts = arrayOfNulls<ASTNode>(ast.size())
                for (i in 0 until ast.size())
                    asts[i] = ast[i]
                Analysis.overallData.addFunction(CustomFun(temp, name, asts, arrayOf()))
                ast.tok = Word(name, Tag.ID)
                ast.clear()
            } else {
                ast.tok = Word.NIL
                ast.clear()
            }
            return ast.tok
        }

        private fun getRandomFunctionName(): String {
            val str = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"
            val builder = StringBuilder()
            val size = ThreadLocalRandom.current().nextInt(6, 12)
            for (i in 0 until size) {
                val number = ThreadLocalRandom.current().nextInt(62)
                builder.append(str[number])
            }
            return builder.toString()
        }

    }

    class CoreToNum : Function("to-num") {
        override fun call(pars: Array<out Token>): Token {
            return BigInteger(pars[0].toString()).toNum()
        }
    }

    class CoreToReal : Function("to-real") {
        override fun call(pars: Array<out Token>): Token {
            return BigDecimal(pars[0].toString()).toReal()
        }
    }

    class CoreToString : Function("to-string") {
        override fun call(pars: Array<out Token>): Token {
            return Word(
                pars[0].toString(),
                Tag.STR
            )
        }
    }

    class CoreReadLine(val context: HimeContext) : Function("read-line") {
        override fun call(pars: Array<out Token>): Token {
            if (Config.GUI) {
                while (GUI.lineFlag.get() == GUI.flag.get());
                GUI.flag.set(GUI.lineFlag.get())
            }
            return Scanner(context.reader).nextLine().toWord()
        }
    }

    class CoreRead(val context: HimeContext) : Function("read") {
        override fun call(pars: Array<out Token>): Token {
            if (Config.GUI) {
                while (GUI.lineFlag.get() == GUI.flag.get());
                GUI.flag.set(GUI.lineFlag.get())
            }
            return Scanner(context.reader).next().toWord()
        }
    }

    class CoreReadNum(val context: HimeContext) : Function("read-num") {
        override fun call(pars: Array<out Token>): Token {
            if (Config.GUI) {
                while (GUI.lineFlag.get() == GUI.flag.get());
                GUI.flag.set(GUI.lineFlag.get())
            }
            return Scanner(context.reader).nextBigInteger().toNum()
        }
    }

    class CoreReadReal(val context: HimeContext) : Function("read-real") {
        override fun call(pars: Array<out Token>): Token {
            if (Config.GUI) {
                while (GUI.lineFlag.get() == GUI.flag.get());
                GUI.flag.set(GUI.lineFlag.get())
            }
            return Scanner(context.reader).nextBigDecimal().toReal()
        }
    }

    class CoreReadBool(val context: HimeContext) : Function("read-bool") {
        override fun call(pars: Array<out Token>): Token {
            if (Config.GUI) {
                while (GUI.lineFlag.get() == GUI.flag.get());
                GUI.flag.set(GUI.lineFlag.get())
            }
            val bool = Scanner(context.reader).nextBoolean()
            return bool.toBool()
        }
    }

    class CoreLen : Function("len") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars[0] is cn.wumoe.hime.lexer.Array)
                (pars[0] as cn.wumoe.hime.lexer.Array).tokens.size.toNum()
            else if (pars[0] is Word)
                pars[0].toString().length.toNum()
            else
                Word.NIL
        }
    }

    class CoreConcat : Function("concat") {
        override fun call(pars: Array<out Token>): Token {
            val builder = StringBuilder()
            for (t in pars)
                builder.append(t.toString())
            return Word(
                builder.toString(),
                Tag.STR
            )
        }
    }

    class CoreGreater : Function(">") {
        override fun call(pars: Array<out Token>): Token {
            if (pars.size >= 2) {
                val parOne: BigDecimal? = if (pars[0] is Num)
                    BigDecimal((pars[0] as Num).value.toString())
                else if (pars[0] is Real)
                    (pars[0] as Real).value
                else
                    return Word.NIL
                val parTwo: BigDecimal? = if (pars[1] is Num)
                    BigDecimal((pars[1] as Num).value.toString())
                else if (pars[1] is Real)
                    (pars[1] as Real).value
                else
                    return Word.NIL
                return if (parOne != null) {
                    if (parOne > parTwo)
                        Word.True
                    else
                        Word.False
                } else
                    Word.NIL
            } else
                return Word.NIL
        }

    }

    class CoreLess : Function("<") {
        override fun call(pars: Array<out Token>): Token {
            if (pars.size >= 2) {
                val parOne: BigDecimal? = if (pars[0] is Num)
                    BigDecimal((pars[0] as Num).value.toString())
                else if (pars[0] is Real)
                    (pars[0] as Real).value
                else
                    return Word.NIL
                val parTwo: BigDecimal? = if (pars[1] is Num)
                    BigDecimal((pars[1] as Num).value.toString())
                else if (pars[1] is Real)
                    (pars[1] as Real).value
                else
                    return Word.NIL
                return if (parOne != null) {
                    if (parOne < parTwo)
                        Word.True
                    else
                        Word.False
                } else
                    Word.NIL
            } else
                return Word.NIL
        }
    }

    class CoreGreaterEq : Function(">=") {
        override fun call(pars: Array<out Token>): Token {
            if (pars.size >= 2) {
                val parOne: BigDecimal? = if (pars[0] is Num)
                    BigDecimal((pars[0] as Num).value.toString())
                else if (pars[0] is Real)
                    (pars[0] as Real).value
                else
                    return Word.NIL
                val parTwo: BigDecimal? = if (pars[1] is Num)
                    BigDecimal((pars[1] as Num).value.toString())
                else if (pars[1] is Real)
                    (pars[1] as Real).value
                else
                    return Word.NIL
                return if (parOne != null) {
                    if (parOne >= parTwo)
                        Word.True
                    else
                        Word.False
                } else
                    Word.NIL
            } else
                return Word.NIL
        }

    }

    class CoreLessEq : Function("<=") {
        override fun call(pars: Array<out Token>): Token {
            if (pars.size >= 2) {
                val parOne: BigDecimal? = if (pars[0] is Num)
                    BigDecimal((pars[0] as Num).value.toString())
                else if (pars[0] is Real)
                    (pars[0] as Real).value
                else
                    return Word.NIL
                val parTwo: BigDecimal? = if (pars[1] is Num)
                    BigDecimal((pars[1] as Num).value.toString())
                else if (pars[1] is Real)
                    (pars[1] as Real).value
                else
                    return Word.NIL
                return if (parOne != null) {
                    if (parOne <= parTwo)
                        Word.True
                    else
                        Word.False
                } else
                    Word.NIL
            } else
                return Word.NIL
        }
    }

    class CoreEq : Function("=") {
        override fun call(pars: Array<out Token>): Token {
            for (i in 1 until pars.size)
                if (pars[i] != pars[i - 1])
                    return Word.False
            return Word.True
        }
    }

    class CoreNotEq : Function("!=") {
        override fun call(pars: Array<out Token>): Token {
            for (i in 1 until pars.size)
                if (pars[i] != pars[i - 1])
                    return Word.True
            return Word.False
        }
    }

    class CoreNot : Function("not") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty() && pars[0] is Word)
                if ((pars[0] as Word).tag == Tag.TRUE)
                    Word.False
                else
                    Word.True
            else
                Word.NIL
        }
    }

    class CoreAnd : Function("and") {
        override fun call(pars: Array<out Token>): Token {
            if (pars.isNotEmpty()) {
                for (t in pars) {
                    if (t is Word) {
                        if (t.tag == Tag.FALSE)
                            return Word.False
                    } else
                        return Word.NIL
                }
                return Word.True
            } else
                return Word.NIL
        }
    }

    class CoreOr : Function("or") {
        override fun call(pars: Array<out Token>): Token {
            if (pars.isNotEmpty()) {
                for (t in pars) {
                    if (t is Word) {
                        if (t.tag == Tag.TRUE)
                            return Word.True
                    } else
                        return Word.NIL
                }
                return Word.False
            } else
                return Word.NIL
        }
    }

    class CoreList : Function("list") {
        override fun call(pars: Array<out Token>): Token {
            return Array(pars)
        }
    }

    class CoreAppend : Function("append") {
        override fun call(pars: Array<out Token>): Token {
            val temp = ArrayList<Token>()
            for (t in pars) {
                if (t is cn.wumoe.hime.lexer.Array)
                    temp.addAll(t.tokens)
                else
                    temp.add(t)
            }
            return Array(temp)
        }
    }

    class CoreCons : Function("cons") {
        override fun call(pars: Array<out Token>): Token {
            return Array(arrayOf(pars[0], pars[1]))
        }
    }

    class CoreCar : Function("car") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars[0] is cn.wumoe.hime.lexer.Array)
                (pars[0] as cn.wumoe.hime.lexer.Array)[0]
            else
                Word.NIL
        }
    }

    class CoreCdr : Function("cdr") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars[0] is cn.wumoe.hime.lexer.Array) {
                val array = ArrayList<Token>()
                array.addAll((pars[0] as cn.wumoe.hime.lexer.Array).tokens)
                array.remove((pars[0] as cn.wumoe.hime.lexer.Array).tokens[0])
                if (array.size == 1)
                    array[0]
                else
                    Array(array)
            } else
                Word.NIL
        }
    }

    class CoreGet : Function("get") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2 && pars[0] is cn.wumoe.hime.lexer.Array && pars[1] is Num)
                (pars[0] as cn.wumoe.hime.lexer.Array).get((pars[1] as Num).value.toInt())
            else
                Word.NIL
        }
    }

    class CoreSet : Function("set") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 3 && pars[0] is cn.wumoe.hime.lexer.Array && pars[1] is Num) {
                (pars[0] as cn.wumoe.hime.lexer.Array).set((pars[1] as Num).value.toInt(), pars[2])
                pars[0]
            } else
                Word.NIL
        }
    }

    class CoreAdd : Function("add") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2 && pars[0] is cn.wumoe.hime.lexer.Array) {
                if (pars.size >= 3) {
                    (pars[0] as cn.wumoe.hime.lexer.Array).add((pars[1] as Num).value.toInt(), pars[2])
                    pars[2]
                } else {
                    (pars[0] as cn.wumoe.hime.lexer.Array).add(pars[1])
                    pars[1]
                }
            } else
                Word.NIL
        }
    }

    class CoreRange : Function("range") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2 && pars[0] is Num && pars[1] is Num) {
                val start = (pars[0] as Num).value
                val end = (pars[1] as Num).value
                val step = if (pars.size >= 3) (pars[2] as Num).value else BigInteger.ONE
                val size = end.subtract(start).divide(step)
                val array = ArrayList<Token>()
                var i = BigInteger.ZERO
                while (i.compareTo(size) != 1) {
                    array.add(start.add(i.multiply(step)).toNum())
                    i = i.add(BigInteger.ONE)
                }
                Array(array)
            } else
                Word.NIL
        }
    }

    class CoreSum : Function("+") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                var num = BigDecimal.ZERO
                for (t in pars)
                    num = num.add(BigDecimal(t.toString()))
                if (!num.simplification().toString().contains("."))
                    BigInteger(num.simplification().toString()).toNum()
                else
                    num.toReal()
            } else
                Word.NIL
        }
    }

    class CoreSub : Function("-") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                var num = BigDecimal(pars[0].toString())
                for (i in 1 until pars.size)
                    num = num.subtract(BigDecimal(pars[i].toString()))
                if (!num.simplification().toString().contains("."))
                    BigInteger(num.simplification().toString()).toNum()
                else
                    num.toReal()
            } else
                Word.NIL
        }
    }

    class CoreDiv : Function("/") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                var num = BigDecimal(pars[0].toString())
                for (i in 1 until pars.size)
                    num = num.divide(BigDecimal(pars[i].toString()), 10, RoundingMode.CEILING)
                if (!num.simplification().toString().contains("."))
                    BigInteger(num.simplification().toString()).toNum()
                else
                    num.toReal()
            } else
                Word.NIL
        }
    }

    class CoreMult : Function("*") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                var num = BigDecimal.ONE
                for (t in pars)
                    num = num.multiply(BigDecimal(t.toString()))
                if (!num.simplification().toString().contains("."))
                    BigInteger(num.simplification().toString()).toNum()
                else
                    num.toReal()
            } else
                Word.NIL
        }
    }

    class Println(val context: HimeContext) : Function("println") {
        override fun call(pars: Array<out Token>): Token {
            val builder = StringBuilder()
            for (token in pars)
                builder.append(token.toString())
            context.writer.write(builder.toString() + "\n")
            context.writer.flush()
            Config.OUT = true
            return Word.NIL
        }
    }

    class Print(val context: HimeContext) : Function("print") {
        override fun call(pars: Array<out Token>): Token {
            val builder = StringBuilder()
            for (token in pars)
                builder.append(token.toString())
            context.writer.write(builder.toString())
            context.writer.flush()
            Config.OUT = true
            return Word.NIL
        }
    }
}