package cn.wumoe.hime.module

import cn.wumoe.hime.api.scripting.HimeContext
import cn.wumoe.hime.inter.Function
import cn.wumoe.hime.inter.Module
import cn.wumoe.hime.lexer.*
import cn.wumoe.hime.toNum
import cn.wumoe.hime.toWord
import kotlin.Array

class StringModule : Module("hime.string") {
    override fun init(context: HimeContext) {
        addFunction(Replace())      // string-replace
        addFunction(Substring())    // string-substring
        addFunction(Split())        // string-split
        addFunction(StringToList()) // string->list
        addFunction(Format())       // string-format
        addFunction(IndexOf())      // string-index
        addFunction(LastIndexOf())  // string-last-index
    }

    class IndexOf : Function("string-index") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2) {
                pars[0].toString()
                    .indexOf(pars[1].toString()).toNum()
            } else
                Word.NIL
        }
    }
    class LastIndexOf : Function("string-last-index") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2) {
                pars[0].toString()
                    .lastIndexOf(pars[1].toString()).toNum()
            } else
                Word.NIL
        }
    }

    class Format : Function("string-format") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2) {
                val args = arrayOfNulls<Any>(pars.size - 1)
                for (i in 0 until pars.size - 1) {
                    when (val token = pars[i + 1]) {
                        is Num -> args[i] = token.value.toLong()
                        is Real -> args[i] = token.value.toDouble()
                        else -> args[i] = token.toString()
                    }
                }
               String.format(pars[0].toString(), *args).toWord()
            } else
                Word.NIL
        }
    }

    class StringToList : Function("string->list") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty() && pars[0] is Word) {
                val chars = pars[0].toString().toCharArray()
                val array = Array(ArrayList())
                for (c in chars)
                    array.add(
                        Word(
                            c.toString(),
                            Tag.STR
                        )
                    )
                array
            } else
                Word.NIL
        }
    }

    class Replace : Function("string-replace") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 3) {
                pars[0].toString()
                    .replace(Regex(pars[1].toString()), pars[2].toString()).toWord()
            } else
                Word.NIL
        }
    }

    class Substring : Function("string-substring") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 3 && pars[1] is Num && pars[2] is Num) {
                pars[0].toString()
                    .substring((pars[1] as Num).value.toInt(), (pars[2] as Num).value.toInt()).toWord()
            } else
                Word.NIL
        }
    }

    class Split : Function("string-split") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2) {
                val result = Array(ArrayList())
                val array = pars[0].toString().split(pars[1].toString())
                for (s in array)
                    result.add(
                        Word(
                            s,
                            Tag.STR
                        )
                    )
                result
            } else
                Word.NIL
        }
    }
}