package cn.wumoe.hime.module

import cn.wumoe.hime.api.scripting.HimeContext
import cn.wumoe.hime.inter.Function
import cn.wumoe.hime.inter.Module
import cn.wumoe.hime.lexer.Token
import cn.wumoe.hime.lexer.Word
import cn.wumoe.hime.toWord
import java.io.File
import java.nio.file.Files
import java.nio.file.Paths
import java.util.*

class FileModule : Module("hime.file") {
    override fun init(context: HimeContext) {
        addFunction(FileExists())           // file-exists
        addFunction(FileList())             // file-list
        addFunction(FileMkdirs())           // file-mkdirs
        addFunction(FileNewFile())          // file-new-file
        addFunction(FileReadBase64())       // file-read-base64
        addFunction(FileWriteBase64())      // file-write-base64
        addFunction(FileReadString())       // file-read-string
        addFunction(FileWriteString())      // file-write-string
    }

    class FileList : Function("file-list") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty())
                listAllFile(File(pars[0].toString()))
            else
                Word.NIL
        }

        private fun listAllFile(f: File) : cn.wumoe.hime.lexer.Array {
            val array = cn.wumoe.hime.lexer.Array(ArrayList())
            val files = f.listFiles()
            for (file in files!!) {
                if (file.isDirectory)
                    array.add(listAllFile(file))
                else
                    array.add(file.path.toWord())
            }
            return array
        }
    }

    class FileExists : Function("file-exists") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                if (File(pars[0].toString()).exists())
                    Word.True
                else
                    Word.False
            } else
                Word.NIL
        }
    }

    class FileMkdirs : Function("file-mkdirs") {
        override fun call(pars: Array<out Token>): Token {
            if (pars.isNotEmpty()) {
                val file = File(pars[0].toString())
                if (!file.parentFile.exists())
                    !file.parentFile.mkdirs()
                if (!file.exists())
                    file.createNewFile()
            }
            return Word.NIL
        }
    }

    class FileNewFile : Function("file-new-file") {
        override fun call(pars: Array<out Token>): Token {
            if (pars.isNotEmpty()) {
                val file = File(pars[0].toString())
                if (!file.exists())
                    file.createNewFile()
            }
            return Word.NIL
        }
    }

    class FileReadBase64 : Function("file-read-base64") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty())
                Base64.getEncoder().encodeToString(Files.readAllBytes(Paths.get(pars[0].toString()))).toWord()
            else
                Word.NIL
        }
    }

    class FileWriteBase64 : Function("file-write-base64") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2) {
                val file = File(pars[0].toString())
                if (!file.parentFile.exists())
                    !file.parentFile.mkdirs()
                if (!file.exists())
                    file.createNewFile()
                Files.write(file.toPath(), Base64.getDecoder().decode(pars[1].toString()))
                pars[0]
            } else
                Word.NIL
        }
    }

    class FileReadString : Function("file-read-string") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty())
                Files.readString(Paths.get(pars[0].toString())).toWord()
            else
                Word.NIL
        }
    }

    class FileWriteString : Function("file-write-string") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 2) {
                val file = File(pars[0].toString())
                if (!file.parentFile.exists())
                    !file.parentFile.mkdirs()
                if (!file.exists())
                    file.createNewFile()
                Files.writeString(file.toPath(), pars[1].toString())
                pars[0]
            } else
                Word.NIL
        }
    }
}