package cn.wumoe.hime.module

import com.google.zxing.*
import com.google.zxing.client.j2se.BufferedImageLuminanceSource
import com.google.zxing.client.j2se.MatrixToImageWriter
import com.google.zxing.common.HybridBinarizer
import cn.wumoe.hime.api.scripting.HimeContext
import cn.wumoe.hime.inter.Function
import cn.wumoe.hime.inter.Module
import cn.wumoe.hime.lexer.Num
import cn.wumoe.hime.lexer.Token
import cn.wumoe.hime.lexer.Word
import cn.wumoe.hime.toWord
import java.io.ByteArrayInputStream
import java.io.ByteArrayOutputStream
import java.util.*
import javax.imageio.ImageIO

class BarcodeModule : Module("hime.barcode") {
    override fun init(context: HimeContext) {
        addFunction(BarcodeCodeEncode())    // barcode-encode
        addFunction(BarcodeCodeDecode())    // barcode-decode
    }

    // (barcode-decode barcode)
    class BarcodeCodeDecode : Function("barcode-decode") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.isNotEmpty()) {
                val image = ImageIO.read(
                    ByteArrayInputStream(Base64.getDecoder().decode(pars[0].toString())))
                val source: LuminanceSource = BufferedImageLuminanceSource(image)
                val bitmap = BinaryBitmap(HybridBinarizer(source))
                val result = MultiFormatReader().decode(bitmap, null)
                result.text.toWord()
            } else
                Word.NIL
        }
    }

    // (barcode-encode width height content)
    class BarcodeCodeEncode : Function("barcode-encode") {
        override fun call(pars: Array<out Token>): Token {
            return if (pars.size >= 3 && pars[0] is Num && pars[1] is Num) {
                val width = (pars[0] as Num).value.toInt()
                val height = (pars[1] as Num).value.toInt()
                val value = pars[2].toString()
                if (checkStandardUPCEANChecksum(value)) {
                    val bitMatrix = MultiFormatWriter().encode(
                        pars[2].toString(),
                        BarcodeFormat.EAN_13, width, height, null
                    )
                    val image = MatrixToImageWriter.toBufferedImage(bitMatrix)
                    val byteArrayOutputStream = ByteArrayOutputStream()
                    ImageIO.write(image, "png", byteArrayOutputStream)
                    Base64.getEncoder().encodeToString(byteArrayOutputStream.toByteArray()).toWord()
                } else
                    Word.NIL
            } else
                Word.NIL
        }

        private fun checkStandardUPCEANChecksum(s: CharSequence): Boolean {
            val length = s.length
            if (length == 0)
                return false
            var sum = 0
            run {
                var i = length - 2
                while (i >= 0) {
                    val digit = s[i].code - '0'.code
                    if (digit < 0 || digit > 9)
                        return false
                    sum += digit
                    i -= 2
                }
            }
            sum *= 3
            var i = length - 1
            while (i >= 0) {
                val digit = s[i].code - '0'.code
                if (digit < 0 || digit > 9)
                    return false
                sum += digit
                i -= 2
            }
            return sum % 10 == 0
        }
    }
}