package cn.wjee.starter.xxl.scanner;

import cn.wjee.commons.lang.StringUtils;
import cn.wjee.commons.lock.DistributedLock;
import cn.wjee.starter.xxl.XxlJobProperties;
import cn.wjee.starter.xxl.annotation.EnableXxlAutoReg;
import cn.wjee.starter.xxl.annotation.XxlAutoReg;
import cn.wjee.starter.xxl.client.XxlAdminClient;
import cn.wjee.starter.xxl.client.request.ExecutorAddRequest;
import cn.wjee.starter.xxl.client.request.JobAddRequest;
import cn.wjee.starter.xxl.client.response.ExecutorPageResponse;
import cn.wjee.starter.xxl.client.response.GroupPageResponse;
import cn.wjee.starter.xxl.client.response.JobPageResponse;
import cn.wjee.starter.xxl.client.response.LoginResponse;
import cn.wjee.starter.xxl.enums.GlueType;
import com.xxl.job.core.handler.annotation.XxlJob;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.SmartInitializingSingleton;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.MethodIntrospector;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.util.ClassUtils;
import org.springframework.util.CollectionUtils;

import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.function.BiConsumer;

/**
 * XxlJob Handler扫描自动注册
 *
 * @author listening
 */
public class XxlJobAutoScanner implements ApplicationContextAware, SmartInitializingSingleton {
    /**
     * 日志
     */
    protected static final Logger log = LoggerFactory.getLogger(XxlJobAutoScanner.class);
    /**
     * Spring Context
     */
    private static ApplicationContext applicationContext;

    private final AtomicBoolean started = new AtomicBoolean(false);

    private final XxlAdminClient xxlAdminClient;

    private final DistributedLock distributedLock;

    private final String appName;

    XxlJobAutoScanner(XxlJobProperties xxlJobProperties, DistributedLock distributedLock, String appName) {
        this.xxlAdminClient = new XxlAdminClient(
            xxlJobProperties.getUrl(),
            xxlJobProperties.getUsername(),
            xxlJobProperties.getPassword()
        );
        this.distributedLock = distributedLock;
        this.appName = appName;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        XxlJobAutoScanner.applicationContext = applicationContext;
    }

    @Override
    public void afterSingletonsInstantiated() {
        boolean lockEnabled = distributedLock != null;
        if (!lockEnabled) {
            log.warn(">>>>> 分布式锁 [DistributedLock] 无可用实现, 默认使用无锁方式注册JOB, 多节点同时启动有重复注册的风险~");
        }

        Lock tempLocked = null;
        if (started.compareAndSet(false, true)) {
            try {
                String lockKey = appName + ":xxl-job:auto-register";
                if (lockEnabled) {
                    tempLocked = distributedLock.lock(lockKey, 5000L, 5 * 60 * 1000L);
                }
                // 有锁 && 锁定失败
                if (lockEnabled && tempLocked == null) {
                    log.info(">>>>>>>>>>> Xxl加锁失败, 忽略自动注册！！！！");
                    return;
                }
                // 有锁 && 锁定成功
                if (lockEnabled) {
                    scanAutoRegXxlJobConfig(applicationContext);
                    return;
                }
                // 无锁
                scanAutoRegXxlJobConfig(applicationContext);
            } catch (Exception e) {
                log.error("XxlJobAutoScanner afterSingletonsInstantiated fail", e);
            } finally {
                if (lockEnabled && tempLocked != null) {
                    distributedLock.unlock(tempLocked);
                }
            }
        }
    }

    /**
     * 扫描自动注册XXL
     *
     * @param applicationContext Context
     */
    private synchronized void scanAutoRegXxlJobConfig(ApplicationContext applicationContext) {
        if (applicationContext == null) {
            return;
        }

        LoginResponse login = xxlAdminClient.login(3);
        if (login == null || !login.isSuccess()) {
            return;
        }

        // init job handler from method
        String[] beanDefinitionNames = applicationContext.getBeanNamesForType(Object.class, false, true);
        boolean groupSuccess = registerGroupAndExecutor(login, beanDefinitionNames);
        if (groupSuccess) {
            scanAndRegisterJob(login, beanDefinitionNames);
        }
    }

    /**
     * 自动注册XXL JOB
     *
     * @param loginResponse       登录信息
     * @param beanDefinitionNames bean列表
     */
    private void scanAndRegisterJob(LoginResponse loginResponse, String[] beanDefinitionNames) {
        try {
            final Map<Method, XxlJob> finalAnnotatedMethods = new HashMap<>();
            for (String beanDefinitionName : beanDefinitionNames) {
                Object bean = applicationContext.getBean(beanDefinitionName);

                Map<Method, XxlJob> annotatedMethods = null;
                try {
                    annotatedMethods = MethodIntrospector.selectMethods(bean.getClass(),
                        (MethodIntrospector.MetadataLookup<XxlJob>) method -> AnnotatedElementUtils.findMergedAnnotation(method, XxlJob.class)
                    );
                } catch (Throwable ignored) {
                }
                if (annotatedMethods == null || annotatedMethods.isEmpty()) {
                    continue;
                }
                finalAnnotatedMethods.putAll(annotatedMethods);
            }

            // 执行器缓存
            final ConcurrentHashMap<String, Integer> executorCacheMap = new ConcurrentHashMap<>();
            for (Map.Entry<Method, XxlJob> methodXxlJobEntry : finalAnnotatedMethods.entrySet()) {
                Method executeMethod = methodXxlJobEntry.getKey();
                XxlJob xxlJob = methodXxlJobEntry.getValue();
                pushXxlJobHandler(loginResponse, executeMethod, xxlJob, executorCacheMap);
            }
        } catch (BeansException ignored) {
        }
    }

    /**
     * 扫描Spring类并注册业务组+执行器
     *
     * @param loginResponse       登录信息
     * @param beanDefinitionNames SpringBean列表
     * @return boolean
     */
    private boolean registerGroupAndExecutor(LoginResponse loginResponse, String[] beanDefinitionNames) {
        try {
            // 同时有SpringBootApplication + EnableXxlJobAuto 注解的类
            final Set<Class<?>> allHasTypes = new LinkedHashSet<>();

            // 只有EnableXxlJobAuto 注解的类
            final Set<Class<?>> onlyHasTypes = new LinkedHashSet<>();

            // 循环Spring中所有Bean定义，找有对应注解的类
            for (String beanDefinitionName : beanDefinitionNames) {
                Object bean = applicationContext.getBean(beanDefinitionName);

                Set<Class<?>> handlerTypes = new LinkedHashSet<>();
                Class<?> specificHandlerType = null;
                if (!Proxy.isProxyClass(bean.getClass())) {
                    specificHandlerType = ClassUtils.getUserClass(bean.getClass());
                    handlerTypes.add(specificHandlerType);
                }
                handlerTypes.addAll(ClassUtils.getAllInterfacesForClassAsSet(bean.getClass()));

                for (Class<?> currentHandlerType : handlerTypes) {
                    final Class<?> targetClass = (specificHandlerType != null ? specificHandlerType : currentHandlerType);
                    boolean xxlAutoPresent = targetClass.isAnnotationPresent(EnableXxlAutoReg.class);
                    boolean springBootPresent = targetClass.isAnnotationPresent(SpringBootApplication.class);
                    if (xxlAutoPresent && springBootPresent) {
                        allHasTypes.add(targetClass);
                    } else if (xxlAutoPresent) {
                        onlyHasTypes.add(targetClass);
                    }
                }
            }

            // 循环筛选的类取一个(优先取共有俩注解的类)
            ArrayList<Class<?>> annotatedClzList = new ArrayList<>(allHasTypes);
            annotatedClzList.addAll(onlyHasTypes);

            Class<?> finalAnnotatedClz = annotatedClzList.stream().findFirst().orElse(null);
            if (finalAnnotatedClz == null) {
                return false;
            }

            // 解析类EnableXxlJobAuto注解
            EnableXxlAutoReg annotation = finalAnnotatedClz.getAnnotation(EnableXxlAutoReg.class);
            if (!annotation.auto()) {
                return true;
            }

            // 注册组 + 执行器 到XxlJob
            String bizGroup = annotation.group();
            String executorName = annotation.executor();
            return pushXxlGroupExecutor(loginResponse, bizGroup, executorName);
        } catch (BeansException ignored) {
            return false;
        }
    }

    /**
     * 推送业务组+执行器到XXL
     *
     * @param loginResponse 登录信息
     * @param group         业务组
     * @param executorName  执行器
     */
    private boolean pushXxlGroupExecutor(LoginResponse loginResponse, String group, String executorName) {
        try {
            AtomicInteger alreadyRegGroupId = new AtomicInteger(0);
            checkGroup(loginResponse, group, (flag, groupId) -> alreadyRegGroupId.set(flag ? groupId : 0));

            List<String> cookie = loginResponse.getCookie();
            // 注册业务组
            if (alreadyRegGroupId.get() <= 0) {
                boolean b = xxlAdminClient.addGroup(cookie, group);
                log.info(">>>>>>>>>>> xxl-job 自动注册业务组 [{}]{}", group, b ? "成功" : "失败!!!!");
                if (!b) {
                    return false;
                }
                checkGroup(loginResponse, group, (flag, groupId) -> alreadyRegGroupId.set(flag ? groupId : 0));
            } else {
                log.info(">>>>>>>>>>> xxl-job 自动注册业务组 [{}], 检测已存在忽略", group);
            }
            if (alreadyRegGroupId.get() <= 0) {
                return false;
            }

            // 注册执行器
            AtomicInteger alreadyRegisterExecutor = new AtomicInteger(0);
            checkExecutor(loginResponse, executorName, (flag, executorId) -> alreadyRegisterExecutor.set(flag ? executorId : 0));
            if (alreadyRegisterExecutor.get() <= 0) {
                ExecutorAddRequest executorAddRequest = new ExecutorAddRequest()
                    .setAppName(appName)
                    .setTitle(executorName)
                    .setBusId(alreadyRegGroupId.get())
                    .setAddressType(0);
                boolean result = xxlAdminClient.addExecutor(cookie, executorAddRequest);
                log.info(">>>>>>>>>>> xxl-job 自动注册执行器 [{}]{}", executorName, result ? "成功" : "失败!!!!");
                if (result) {
                    checkExecutor(loginResponse, executorName, (flag, executorId) -> alreadyRegisterExecutor.set(flag ? executorId : 0));
                }
            } else {
                log.info(">>>>>>>>>>> xxl-job 自动注册执行器 [{}], 检测已存在忽略", executorName);
            }
            return alreadyRegisterExecutor.get() > 0;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 检测Group是否创建成功
     *
     * @param loginResponse 登录信息
     * @param group         业务组名称
     * @param consumer      回调(是否成功, 业务组ID)
     */
    private void checkGroup(LoginResponse loginResponse, String group, BiConsumer<Boolean, Integer> consumer) {
        try {
            List<String> cookie = loginResponse.getCookie();
            int groupId = 0;
            boolean hasAddGroup = false;
            final int size = 100;
            int pageNo = 1;
            GroupPageResponse groupPage;
            do {
                groupPage = xxlAdminClient.getGroupPage(cookie, pageNo, size);
                List<GroupPageResponse.GroupInfo> groupInfos = groupPage != null ? groupPage.getData() : null;
                GroupPageResponse.GroupInfo groupInfo = Optional.ofNullable(groupInfos)
                    .orElse(new ArrayList<>())
                    .stream()
                    .filter(g -> g.getName().equals(group))
                    .findFirst().orElse(null);
                if (groupInfo != null) {
                    hasAddGroup = true;
                    groupId = groupInfo.getId();
                } else {
                    pageNo += 1;
                }
            } while (groupPage != null && groupPage.getData() != null && groupPage.getData().size() == size);

            consumer.accept(hasAddGroup, groupId);
        } catch (Exception e) {
            consumer.accept(false, null);
        }
    }

    /**
     * 检测执行器注册结果
     *
     * @param loginResponse 登录信息
     * @param executorName  执行器名称
     * @param consumer      回调(是否创建成功,执行器ID)
     */
    private void checkExecutor(LoginResponse loginResponse, String executorName, BiConsumer<Boolean, Integer> consumer) {
        ExecutorPageResponse executorPage = xxlAdminClient.getExecutorPage(loginResponse.getCookie(), appName);

        List<ExecutorPageResponse.ExecutorInfo> executorList = new ArrayList<>();
        if (executorPage != null && executorPage.getData() != null) {
            executorList = executorPage.getData();
        }
        ExecutorPageResponse.ExecutorInfo executorInfo = executorList
            .stream()
            .filter(s -> s.getAppName().equals(appName))
            .filter(s -> s.getTitle().equals(executorName))
            .findFirst().orElse(null);
        consumer.accept(executorInfo != null, executorInfo != null ? executorInfo.getId() : 0);
    }

    /**
     * 推送任务到XXL
     *
     * @param loginResponse    登录信息
     * @param executeMethod    方法
     * @param xxlJob           XXL JOB
     * @param executorCacheMap 执行器缓存
     */
    private void pushXxlJobHandler(LoginResponse loginResponse, Method executeMethod, XxlJob xxlJob, ConcurrentHashMap<String, Integer> executorCacheMap) {
        try {
            if (!executeMethod.isAnnotationPresent(XxlAutoReg.class)) {
                return;
            }
            XxlAutoReg xxlAutoReg = executeMethod.getAnnotation(XxlAutoReg.class);
            String executorName = xxlAutoReg.executor();
            if (StringUtils.isAnyBlank(executorName, xxlAutoReg.jobName())) {
                return;
            }

            Integer executorId = executorCacheMap.get(executorName);
            if (executorId == null) {
                AtomicInteger registeredExecutorId = new AtomicInteger(0);
                checkExecutor(loginResponse, executorName, (flag, tempExecutorId) -> registeredExecutorId.set(flag ? tempExecutorId : 0));
                executorCacheMap.put(executorName, registeredExecutorId.get());
                executorId = executorCacheMap.get(executorName);
            }
            if (executorId == null || executorId <= 0) {
                return;
            }

            final List<JobPageResponse.JobInfo> jobInfoList = new ArrayList<>();
            int pageNo = 1;
            while (true) {
                JobPageResponse jobPage = xxlAdminClient.getJobPage(loginResponse.getCookie(), executorId, null, pageNo, 100);
                List<JobPageResponse.JobInfo> tempJobInfoList = jobPage != null ? jobPage.getData() : new ArrayList<>();
                List<JobPageResponse.JobInfo> addedJobList = Optional.ofNullable(tempJobInfoList).orElse(new ArrayList<>());

                if (CollectionUtils.isEmpty(addedJobList) || addedJobList.size() < 100) {
                    jobInfoList.addAll(addedJobList);
                    break;
                }
                jobInfoList.addAll(addedJobList);
                pageNo += 1;
            }

            // 注册JOB
            boolean processReg;
            GlueType glueType = xxlAutoReg.glueType();
            if (glueType == GlueType.BEAN) {
                String jobHandler = xxlJob.value();
                processReg = jobInfoList.stream().noneMatch(s -> s.getExecutorHandler().equals(jobHandler));
            } else {
                processReg = jobInfoList.stream().noneMatch(s -> s.getJobDesc().equals(xxlAutoReg.jobName()));
            }
            if (!processReg) {
                log.info(">>>>>>>>>>> xxl-job 自动注册任务 [{}], 检测已存在忽略", xxlAutoReg.jobName());
                return;
            }
            JobAddRequest jobAddRequest = new JobAddRequest()
                .setJobExecutor(executorId)
                .setJobDesc(xxlAutoReg.jobName())
                .setAuthor(xxlAutoReg.author())
                .setAlarmEmail(xxlAutoReg.email())
                .setScheduleType(xxlAutoReg.scheduleType())
                .setScheduleConf(xxlAutoReg.corn())
                .setGlueType(xxlAutoReg.glueType())
                .setGlueSource(xxlAutoReg.glueSource())
                .setGlueRemark(xxlAutoReg.glueRemark())
                .setExecutorHandler(xxlJob.value())
                .setExecutorParam(xxlAutoReg.jobParam())
                .setExecutorRouteStrategy(xxlAutoReg.routeType())
                .setMisfireStrategy(xxlAutoReg.misFireStrategy())
                .setExecutorBlockStrategy(xxlAutoReg.blockStrategy())
                .setChildJobId("")
                .setExecutorTimeout(xxlAutoReg.executorTimeout())
                .setExecutorFailRetryCount(xxlAutoReg.executorFailRetryCount());

            if (xxlAdminClient.addJob(loginResponse.getCookie(), jobAddRequest)) {
                log.info(">>>>>>>>>>> xxl-job 自动注册任务 [{}]成功", jobAddRequest.getJobDesc());
                JobPageResponse.JobInfo jobInfo = new JobPageResponse.JobInfo();
                jobInfo.setJobDesc(xxlAutoReg.jobName());
                jobInfo.setExecutorHandler(glueType == GlueType.BEAN ? xxlJob.value() : "");
                jobInfoList.add(jobInfo);
            } else {
                log.info(">>>>>>>>>>> xxl-job 自动注册任务 [{}]失败！！！！", jobAddRequest.getJobDesc());
            }
        } catch (Exception ignored) {
        }
    }
}
