package cn.wjee.starter.xxl.api;

import cn.wjee.commons.collection.CollectionUtils;
import cn.wjee.commons.enums.ApiStatusEnum;
import cn.wjee.commons.extra.lang.JacksonUtils;
import cn.wjee.commons.lang.StringUtils;
import cn.wjee.starter.xxl.api.request.ExecutorAddRequest;
import cn.wjee.starter.xxl.api.request.JobAddRequest;
import cn.wjee.starter.xxl.api.response.*;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.*;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.util.List;

/**
 * XXL API
 *
 * @author listening
 */
@Slf4j
public class XxlAdminApi {
    /**
     * 管理地址
     */
    private final String adminUrl;
    /**
     * 认证账号
     */
    private final String username;
    private final String password;
    /**
     * Http客户端
     */
    private final RestTemplate restTemplate = getRestTemplate();

    private static RestTemplate getRestTemplate() {
        SimpleClientHttpRequestFactory simpleClientHttpRequestFactory = new SimpleClientHttpRequestFactory();
        simpleClientHttpRequestFactory.setConnectTimeout(1000);
        simpleClientHttpRequestFactory.setReadTimeout(3000);
        return new RestTemplate(simpleClientHttpRequestFactory);
    }

    /**
     * 构建XxlAdmin客户端
     *
     * @param adminUrl 管理地址
     * @param username 账号
     * @param password 密码
     */
    public XxlAdminApi(String adminUrl, String username, String password) {
        this.adminUrl = adminUrl;
        this.username = username;
        this.password = password;
    }

    /**
     * 登录XXL，在Token缺失的情况调用
     *
     * @param tryTimes 重试次数
     * @return LoginResponse
     */
    public LoginResponse login(int tryTimes) {
        try {
            LoginResponse response = new LoginResponse();
            if (tryTimes < 1) {
                return response;
            }
            String url = adminUrl + "/login";
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);

            MultiValueMap<String, String> params = new LinkedMultiValueMap<>();
            params.add("userName", username);
            params.add("password", password);
            HttpEntity<MultiValueMap<String, String>> requestEntity = new HttpEntity<>(params, headers);

            ResponseEntity<String> exchange = restTemplate.exchange(url, HttpMethod.POST, requestEntity, String.class);
            String respBody = exchange.getBody();
            XxlResponse<Void> xxlResponse = JacksonUtils.fromJson(respBody, new TypeReference<XxlResponse<Void>>() {
            });
            if (xxlResponse == null || !xxlResponse.isSuccess()) {
                return response;
            }
            List<String> cookies = exchange.getHeaders().get("Set-Cookie");
            response.setCookie(cookies);
            response.setSuccess(true);
            return response;
        } catch (Exception e) {
            return login(--tryTimes);
        }
    }

    /**
     * 业务组查询
     *
     * @param cookie   COOKIE
     * @param pageNo   页码
     * @param pageSize 页面大小
     * @return GroupPageResponse
     */
    public GroupPageResponse getGroupPage(List<String> cookie, Integer pageNo, Integer pageSize) {
        try {
            MultiValueMap<String, Object> params = new LinkedMultiValueMap<>();
            params.add("start", (pageNo - 1) * pageSize);
            params.add("length", pageSize);
            final String respBody = postForm(adminUrl + "/jobbusgroup/pageList", params, cookie);
            return JacksonUtils.fromJson(respBody, new TypeReference<GroupPageResponse>() {
            });
        } catch (Exception e) {
            return null;
        }
    }


    /**
     * 查询执行器列表
     *
     * @param cookie  Cookie
     * @param appName 项目名称
     * @return ExecutorPageResponse
     */
    public ExecutorPageResponse getExecutorPage(List<String> cookie, String appName) {
        try {
            MultiValueMap<String, Object> params = new LinkedMultiValueMap<>();
            params.add("appname", appName);
            params.add("title", "");
            params.add("start", 0);
            params.add("length", 10);
            final String respBody = postForm(adminUrl + "/jobgroup/pageList", params, cookie);
            return JacksonUtils.fromJson(respBody, new TypeReference<ExecutorPageResponse>() {
            });
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 查询Job列表
     *
     * @param cookie     Cookie
     * @param executorId 执行器ID
     * @param jobHandler JobBean名称
     * @return JobPageResponse
     */
    public JobPageResponse getJobPage(List<String> cookie, Integer executorId, String jobHandler, Integer pageNo, Integer pageSize) {
        try {
            MultiValueMap<String, Object> params = new LinkedMultiValueMap<>();
            params.add("jobGroup", executorId != null && executorId >= 0 ? executorId : 0);
            params.add("executorHandler", StringUtils.getValue(jobHandler));
            params.add("start", (pageNo - 1) * pageSize);
            params.add("length", pageSize);
            final String respBody = postForm(adminUrl + "/jobinfo/pageList", params, cookie);
            return JacksonUtils.fromJson(respBody, new TypeReference<JobPageResponse>() {
            });
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 添加业务组
     *
     * @param cookie    Cookie参数
     * @param groupName 业务组名
     * @return boolean
     */
    public boolean addGroup(List<String> cookie, String groupName) {
        try {
            MultiValueMap<String, Object> params = new LinkedMultiValueMap<>();
            params.add("name", groupName);
            final String s = postForm(adminUrl + "/jobbusgroup/add", params, cookie);
            final JsonNode root = JacksonUtils.fromJson(s);
            final String code = JacksonUtils.getChildNodeStringValue(root, "code");
            return ApiStatusEnum.SUCCESS.getStringCode().equals(code);
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 添加执行器
     *
     * @param cookie  Cookie参数
     * @param request 执行器创建请求
     * @return boolean
     */
    public boolean addExecutor(List<String> cookie, ExecutorAddRequest request) {
        try {
            final String s = postForm(adminUrl + "/jobgroup/save", request.toMap(), cookie);
            final JsonNode root = JacksonUtils.fromJson(s);
            final String code = JacksonUtils.getChildNodeStringValue(root, "code");
            return ApiStatusEnum.SUCCESS.getStringCode().equals(code);
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 添加执行器
     *
     * @param cookie  Cookie参数
     * @param request 执行器创建请求
     * @return boolean
     */
    public boolean addJob(List<String> cookie, JobAddRequest request) {
        try {
            final String s = postForm(adminUrl + "/jobinfo/add", request.toMap(), cookie);
            final JsonNode root = JacksonUtils.fromJson(s);
            final String code = JacksonUtils.getChildNodeStringValue(root, "code");
            return ApiStatusEnum.SUCCESS.getStringCode().equals(code);
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 接口请求
     *
     * @param url    地址
     * @param params 参数
     * @param cookie COOKIE
     * @return String
     */
    private String postForm(String url, MultiValueMap<String, Object> params, List<String> cookie) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
        if (CollectionUtils.isNotEmpty(cookie)) {
            headers.add(HttpHeaders.COOKIE, StringUtils.join(cookie, ";"));
        }

        HttpEntity<MultiValueMap<String, Object>> requestEntity = new HttpEntity<>(params, headers);
        ResponseEntity<String> exchange = restTemplate.exchange(url, HttpMethod.POST, requestEntity, String.class);
        // System.err.println(body);
        return exchange.getBody();
    }

    //    public static void main(String[] args) {
    //        XxlAdminApi admin = new XxlAdminApi("http://xxl-dev.xxx.com.cn/xxl-job-admin", "admin", "123456");
    //        final LoginResponse login = admin.login(3);
    //        // final ExecutorPageResponse executors = admin.getExecutors(login.getCookie(), "analysis-async-service");
    //        // final JobPageResponse jobPage = admin.getJobPage(login.getCookie(), null, null, 1, 20);
    //        // System.err.println(JacksonUtils.toJson(jobPage));
    //
    //        // final boolean test123 = admin.addGroup(login.getCookie(), "test123");
    //        // System.err.println(test123);
    //
    //        final GroupPageResponse groupPage = admin.getGroupPage(login.getCookie(), 1, 100);
    //        final Integer groupId = groupPage.getData().stream().filter(s -> s.getName().equals("test123")).findFirst().orElse(new GroupPageResponse.GroupInfo()).getId();
    //        // System.err.println(groupId);
    //
    //        // final boolean b = admin.addExecutor(login.getCookie(), new ExecutorAddRequest().setAppname("test").setTitle("测试").setBusId(groupId));
    //        // System.err.println(b);
    //
    //        final ExecutorPageResponse test = admin.getExecutorPage(login.getCookie(), "test");
    //        final Integer executorId = test.getData().get(0).getId();
    //        // System.err.println(executorId);
    //
    //        //
    //        final JobPageResponse.JobInfo jobInfo = admin
    //            .getJobPage(login.getCookie(), executorId, "", 1, 100)
    //            .getData()
    //            .stream()
    //            .filter(s -> s.getJobDesc().equals("test"))
    //            .findFirst().orElse(new JobPageResponse.JobInfo());
    //        System.err.println(jobInfo.getId());
    //
    //        // Add
    //        JobAddRequest jobAddRequest = new JobAddRequest()
    //            .setJobExecutor(executorId).setJobDesc("ceshi")
    //            .setAuthor("lee").setAlarmEmail("lxn@gongchangtemai.com.cn")
    //            .setScheduleType(ScheduleType.CRON)
    //            .setScheduleConf("0 0/10 * * * ?")
    //            .setGlueType(GlueType.GLUE_GROOVY)
    //            .setGlueSource("package com.xxl.job.service.handler;\n" +
    //                "\n" +
    //                "import com.xxl.job.core.context.XxlJobHelper;\n" +
    //                "import com.xxl.job.core.handler.IJobHandler;\n" +
    //                "\n" +
    //                "public class DemoGlueJobHandler extends IJobHandler {\n" +
    //                "\n" +
    //                "\t@Override\n" +
    //                "\tpublic void execute() throws Exception {\n" +
    //                "\t\tXxlJobHelper.log(\"XXL-JOB, Hello World.\");\n" +
    //                "\t}\n" +
    //                "\n" +
    //                "}\n")
    //            .setExecutorHandler("test123")
    //            .setExecutorParam("param")
    //            .setExecutorRouteStrategy(RouteType.FIRST)
    //            .setMisfireStrategy(MisFireStrategy.DO_NOTHING)
    //            .setExecutorBlockStrategy(BlockStrategy.SERIAL_EXECUTION);
    //
    //        boolean b = admin.addJob(login.getCookie(), jobAddRequest);
    //        System.err.println(b);
    //
    //    }
}
