/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache license, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the license for the specific language governing permissions and
 * limitations under the license.
 */
package cn.wi4j.security.core.handler;

import cn.wi4j.security.core.Wic4jUserDetails;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.StringUtils;

import java.util.Collection;
import java.util.List;

/**
 * 全检校验处理<br>
 *
 * @author Max
 * @version 2023.0.0.0
 * @since 2023/7/11 19:41
 */
@Component(value = "wj")
public class Wic4jPermissionHandler {

    /**
     * 权限匹配
     */
    @Autowired
    @SuppressWarnings("all")
    private AntPathMatcher antPathMatcher;


    /**
     * 验证用户是否具备某权限
     *
     * @param permission 权限字符串
     * @return 用户是否具备某权限
     */
    public boolean hasPerm(String permission) {
        if (!StringUtils.hasText(permission)) {
            return false;
        }
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        Wic4jUserDetails principal = (Wic4jUserDetails) authentication.getPrincipal();
        Collection<? extends GrantedAuthority> authorities = principal.getAuthorities();
        return hasPermissions(authorities.stream().map(GrantedAuthority::getAuthority).toList(), permission);
    }

    /**
     * 校验权限
     *
     * @param permissions 当前用户所具有对的所有的权限
     * @param permission  当前访问的权限
     * @return 是否有权限
     */
    private boolean hasPermissions(List<String> permissions, String permission) {
        for (String per : permissions) {
            if (antPathMatcher.match(per, permission)) {
                return true;
            }
        }
        return false;
    }

}
