package cn.warpin.core.util;

import cn.warpin.core.base.baseEntity.BaseEntity;
import cn.warpin.core.base.jpa.jpaComment.annotation.ColumnComment;
import cn.warpin.core.base.jpa.jpaComment.annotation.TableComment;
import cn.warpin.core.constant.StandardName;
import cn.warpin.core.exception.ResultException;
import cn.warpin.core.result.ResCode;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.google.gson.Gson;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Date;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.stream.Collectors;

public class ObjectUtil {
    /**
     * map封装为对象
     *
     * @param map
     */
    public static void transMap2Bean(LinkedHashMap map, Object obj) {
        try {
            BeanInfo beanInfo = Introspector.getBeanInfo(obj.getClass());
            PropertyDescriptor[] propertyDescriptors = beanInfo.getPropertyDescriptors();
            Class<?> clz = obj.getClass();
            // 获取实体类的所有属性，返回Field数组
            Field[] fields = clz.getDeclaredFields();

            for (PropertyDescriptor property : propertyDescriptors) {
                String key = property.getName();
                Object value = null;
                for (Field field : fields) {
                    if (key.equals(field.getName())) {
                        value = map.get(key);
                        // 如果类型是Long
                        if (field.getGenericType().toString().equals("class java.lang.Long")) {
                            value = Long.valueOf(value.toString());
                        }

                        if (field.getGenericType().toString().equals("class java.lang.Integer")) {
                            value = Integer.valueOf(value.toString());
                        }

                        if (field.getGenericType().toString().equals("class java.sql.Timestamp")) {
                            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss "); // 格式化时间
                            Date timeDate = new Date(Long.parseLong(value.toString()));// util类型
                            java.sql.Timestamp dateTime = new java.sql.Timestamp(timeDate.getTime());// Timestamp类型,timeDate.getTime()返回
                            value = dateTime;
                        }
                    }

                }

                if (map.containsKey(key)) {

                    // 得到property对应的setter方法
                    Method setter = property.getWriteMethod();
                    setter.invoke(obj, value);
                }
            }
        } catch (Exception e) {
            System.out.println("transMap2Bean Error " + e);
        }
    }

    /**
     * 转换map
     *
     * @param originalMap
     * @return
     */
    public static MultiValueMap<String, String> convertMap(Map<String, Object> originalMap) {
        MultiValueMap<String, String> multiValueMap = new LinkedMultiValueMap<>();
        for (Map.Entry<String, Object> entry : originalMap.entrySet()) {
            if (entry.getValue() == null) {
                continue;
            }
            multiValueMap.add(entry.getKey(), entry.getValue().toString());
        }
        return multiValueMap;
    }

    // 将类转换成MultiValueMap
    public static MultiValueMap<String, String> transBean2MultiValueMap(Object obj) {
        Map<String, Object> map = transBean2Map(obj);
        return convertMap(map);
    }

    // 将类转换成Map
    public static Map<String, Object> transBean2Map(Object obj) {

        if (obj == null) {
            return null;
        }

        // 如果obj已经是Map，这里直接返回
        if (obj instanceof Map) {
            return (Map<String, Object>) obj;
        }


        Map<String, Object> res = new HashMap<>();
        String name = "";
        String upperName = "";

        try {
            Field[] fields = getAllFields(obj.getClass());
            for (int i = 0; i < fields.length; i++) {
                name = fields[i].getName(); // 获取属性的名字
                upperName = name.substring(0, 1).toUpperCase() + name.substring(1); // 将属性的首字符大写，方便构造get方法
                Method getMethod = null;// 获取原生类的get方法
                getMethod = obj.getClass().getMethod("get" + upperName);
                Object value = getMethod.invoke(obj);
                if (null != value) {
                    res.put(name, value);
                }
            }
        } catch (InvocationTargetException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        } catch (NoSuchMethodException e) {
            throw new RuntimeException(e);
        }
        return res;
    }


    /**
     * 此方法是使用父类数据为子类赋值
     *
     * @param parentObject 父类对象
     * @param voClass      子类的类型
     * @return
     * @throws SecurityException
     * @throws NoSuchMethodException
     */
    public static Object constructExtendClass(Object parentObject, Class voClass) throws Exception {
        Object result = voClass.newInstance();
        String name = "";
        String upperName = "";
        List<Field> fieldList = new ArrayList<>();
        Class tempClass = parentObject.getClass();
        while (tempClass != null) {//当父类为null的时候说明到达了最上层的父类(Object类).
            fieldList.addAll(Arrays.asList(tempClass.getDeclaredFields()));
            tempClass = tempClass.getSuperclass(); //得到父类,然后赋给自己
        }

        Field[] field = fieldList.toArray(new Field[fieldList.size()]);
        for (int i = 0; i < field.length; i++) {
            name = field[i].getName(); // 获取属性的名字
            upperName = name.substring(0, 1).toUpperCase() + name.substring(1); // 将属性的首字符大写，方便构造get方法
            Method getMethod = parentObject.getClass().getMethod("get" + upperName);// 获取原生类的get方法
            Class type = field[i].getType();
            Object value = getMethod.invoke(parentObject);
            Method setMethod = result.getClass().getMethod("set" + upperName, type);// 获取原生类的get方法
            setMethod.invoke(result, value);
        }
        return result;
    }

    /**
     * 子类赋值给父类
     *
     * @param childObj  子类
     * @param parentObj 父类类型
     * @return
     * @throws SecurityException
     * @throws NoSuchMethodException
     */
    public static Object constructParentClass(Object childObj, Object parentObj) throws Exception {
        String name = "";
        String upperName = "";

        List<Field> fieldList = new ArrayList<>();
        Class tempClass = parentObj.getClass();
        while (tempClass != null) {//当父类为null的时候说明到达了最上层的父类(Object类).
            fieldList.addAll(Arrays.asList(tempClass.getDeclaredFields()));
            tempClass = tempClass.getSuperclass(); //得到父类,然后赋给自己
        }

        Field[] field = fieldList.toArray(new Field[fieldList.size()]);
        for (int i = 0; i < field.length; i++) {
            name = field[i].getName(); // 获取属性的名字
            upperName = name.substring(0, 1).toUpperCase() + name.substring(1); // 将属性的首字符大写，方便构造get方法
            Method getMethod = childObj.getClass().getMethod("get" + upperName);// 获取原生类的get方法
            Class type = field[i].getType();
            Object value = getMethod.invoke(childObj);
            Method setMethod = parentObj.getClass().getMethod("set" + upperName, type);// 获取原生类的get方法
            setMethod.invoke(parentObj, value);
        }

        return parentObj;
    }

    /**
     * 非继承关系类进行赋值  仅适用单层继承
     *
     * @param fromObj 数据来源类
     * @param toObj   数据结果类
     * @return
     * @throws Exception
     */
    public static Object constructParallelClass(Object fromObj, Object toObj) throws Exception {
        String name = "";
        String upperName = "";
        Field[] toField = toObj.getClass().getDeclaredFields();
        for (int i = 0; i < toField.length; i++) {
            name = toField[i].getName(); // 获取结果类属性的名字
            boolean haveField = false;
            Field[] fromField = fromObj.getClass().getDeclaredFields();// 数据来源类
            for (int j = 0; j < fromField.length; j++) {// 判断是否具有该属性
                if (name.equals(fromField[i].getName())) {
                    haveField = true;
                    break;
                }
            }
            if (haveField) {
                upperName = name.substring(0, 1).toUpperCase() + name.substring(1); // 将属性的首字符大写，方便构造get方法
                Method getMethod = fromObj.getClass().getMethod("get" + upperName);// 获取来源类的get方法
                Class type = toField[i].getType();
                Object value = getMethod.invoke(fromObj);// 获取来源类的属性值
                Method setMethod = toObj.getClass().getMethod("set" + upperName, type);// 获取原生类的get方法
                setMethod.invoke(toObj, value);
            }
        }

        return toObj;
    }

    /**
     * 获取查询条件需要的子集和类型
     *
     * @param entityCondition
     * @return
     */
    public static Map<String, Object> getVOClassMap(Object entityCondition) {
        Map<String, Object> result = new HashMap<>();
        String packagePrefix = "cn.warpin.business"; // 要查找的包前缀
        String nameWithPkg = "";
        try {
            Field[] fields = entityCondition.getClass().getDeclaredFields();
            for (int i = 0; i < fields.length; i++) {
                nameWithPkg = fields[i].getName(); // 获取属性的名字
                if (nameWithPkg.startsWith(StandardName.REFLEXPKG)) {
                    Map<String, Object> resValue = new HashMap<>();
                    Method getMethod = entityCondition.getClass().getMethod("get" + GenUtils.upperCaseFirstName(nameWithPkg));// 获取来源类的get方法
                    Object value = getMethod.invoke(entityCondition);
                    if (null == value) {
                        continue;
                    }

                    // 先通过类名找到 value 是表名，而不是列名
                    String className = GenUtils.transformTableName(value.toString(), true);
                    List<Class<?>> matchingClasses = Arrays.stream(Package.getPackages())
                            .map(Package::getName)
                            .filter(packageName -> packageName.startsWith(packagePrefix))
                            .map(packageName -> {
                                try {
                                    return Class.forName(packageName + "." + className + "VO");
                                } catch (ClassNotFoundException e) {
                                    return null;
                                }
                            })
                            .filter(clazz -> clazz != null && BaseEntity.class.isAssignableFrom(clazz))
                            .collect(Collectors.toList());

                    if (ArrayUtil.isEmpty(matchingClasses)) {
                        continue;
                    }
                    Class<?> clz = matchingClasses.get(0);
                    // 获取vo中对应的属性名称（一般是集合）
                    String propertyName = nameWithPkg.substring(StandardName.REFLEXPKG.length());
                    String[] propertyNameArr = propertyName.split("_");
                    String keyName = propertyNameArr[0];
                    resValue.put(keyName, clz);
                    resValue.put("pkg_val", value);
                    result.put(nameWithPkg, resValue);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }


    public static void copyMapPropertiesToClass(Map<String, Object> origin, Object target) {

        if (origin == null) {
            return;
        }
        String nameWithPkg = "";
        try {
            Field[] fields = ObjectUtil.getAllFields(target.getClass());
            for (Field field : fields) {
                // 获取字段的修饰符
                int modifiers = field.getModifiers();
                // 如果是final的，则不赋值
                if (Modifier.isFinal(modifiers)) {
                    continue;
                }
                nameWithPkg = field.getName(); // 获取属性的名字
                Object value = origin.get(nameWithPkg);
                if (value == null) {
                    continue;
                }
                Class<?> type = field.getType();
                value = parseObjectValue(type, value);
                Method setMethod = target.getClass().getMethod("set" + GenUtils.upperCaseFirstName(nameWithPkg), type);// 获取原生类的get方法
                setMethod.invoke(target, value);
            }
        } catch (Exception e) {
            throw new ResultException(ResCode.SERVER_COMMON_ERROR);
        }
    }


    public static Field[] getAllFields(Class<?> clazz) {
        List<Field> fieldList = new ArrayList<>();
        while (clazz != null) {
            fieldList.addAll(new ArrayList<>(Arrays.asList(clazz.getDeclaredFields())));
            clazz = clazz.getSuperclass();
        }
        Field[] fields = new Field[fieldList.size()];
        return fieldList.toArray(fields);
    }


    /**
     * 获取属性值
     *
     * @param object
     * @param key
     * @return
     * @throws Exception
     */
    public static Object getPropertyValue(Object object, String key) {
        String name = "";
        String upperName = "";
        Object value = null;
        Field[] fields = getAllFields(object.getClass());
        try {
            for (int i = 0; i < fields.length; i++) {
                name = fields[i].getName(); // 获取属性的名字
                if (name.equals(key)) {
                    upperName = name.substring(0, 1).toUpperCase() + name.substring(1); // 将属性的首字符大写，方便构造get方法
                    Method getMethod = object.getClass().getMethod("get" + upperName);// 获取原生类的get方法
                    value = getMethod.invoke(object);
                    break;
                }
            }
        } catch (Exception e) {
            throw new ResultException(ResCode.SERVER_COMMON_ERROR);
        }

        return value;
    }

    /**
     * 获取需要的字符串类型的属性值
     *
     * @param object
     * @param keys
     * @return
     * @throws Exception
     */
    public static List<String> getPropertyValues(Object object, String... keys) throws Exception {
        List<String> resultList = new ArrayList<>();
        String name = "";
        String upperName = "";
        Object value = null;
        Field[] field = object.getClass().getDeclaredFields();
        for (int i = 0; i < field.length; i++) {
            name = field[i].getName(); // 获取属性的名字
            for (String key : keys) {
                if (name.equals(key)) {
                    upperName = name.substring(0, 1).toUpperCase() + name.substring(1); // 将属性的首字符大写，方便构造get方法
                    Method getMethod = object.getClass().getMethod("get" + upperName);// 获取原生类的get方法
                    value = getMethod.invoke(object);
                    if (value != null && StrUtil.isNotEmpty(value.toString())) {
                        resultList.add(value.toString());
                    }
                    break;
                }
            }
        }
        return resultList;
    }

    /**
     * 将list<?>转成 List<map>
     *
     * @param list
     * @return
     */
    public static List<Map<String, Object>> toListMap(List<?> list) {
        List<Map<String, Object>> result = new ArrayList<>();
        for (Object o : list) {
            Map<String, Object> map = JSONObject.parseObject(JSONObject.toJSONString(o), new TypeReference<Map<String, Object>>() {
            });
            result.add(map);
        }
        return result;
    }


    // 将对象转成json字符串
    public static String tojson(Object obj) {
        if (null == obj) {
            return null;
        }
        Gson gson = new Gson();
        String jsonStr = gson.toJson(obj);
        return jsonStr;
    }

    // 将json字符串转成对象
    public static Object toObject(String jsonStr, Class clazz) {
        if (StrUtil.isEmpty(jsonStr)) {
            return null;
        }
        Gson gson = new Gson();
        Object resObj = gson.fromJson(jsonStr, clazz);
        return resObj;
    }


    public static List typedList(List untypedList, Class itemClass, Class listClass) {
        List list = null;
        try {
            list = (List) listClass.newInstance();
        } catch (Exception e) {
            e.printStackTrace();
        }
        for (Object item : untypedList) {
            list.add(itemClass.cast(item));
        }
        return list;
    }


    // 对象锁
    private static Map<Integer, Integer> LOCKS_UPDATE = new ConcurrentHashMap<>();

    public static Object getLockByPrimaryKey(final Integer id) {
        LOCKS_UPDATE.putIfAbsent(id, id);
        return LOCKS_UPDATE.get(id);
    }

    public static void unLockByPrimaryKey(final Integer id) {
        LOCKS_UPDATE.remove(id);
    }

    /**
     * 将字符串转换为对应的类型
     *
     * @param type
     * @param value
     * @return
     */
    private static Object parseObjectValue(Class<?> type, Object value) {
        if (value == null) {
            return null;
        }
        if (type == Timestamp.class) {
            value = DateUtil.stringToTimestamp(value.toString());
        }

        if (type == Date.class) {
            value = DateUtil.stringToSqlDate(value.toString(), DateUtil.FORMATTYPE_5);
        }

        if (type == Integer.class) {
            value = Integer.parseInt(value.toString());
        }

        if (type == Long.class) {
            value = Long.parseLong(value.toString());
        }

        if (type == Double.class) {
            value = Double.parseDouble(value.toString());
        }

        if (type == BigDecimal.class) {
            value = new BigDecimal(value.toString());
        }
        return value;
    }

    /**
     * 类查找器
     *
     * @param targetClass
     * @return
     * @throws IOException
     */
    public static String classFinder(String targetClass) {
        try {
            Enumeration<URL> roots = Thread.currentThread().getContextClassLoader().getResources("");
            while (roots.hasMoreElements()) {
                URL url = roots.nextElement();
                File file = new File(url.getPath());
                if (file.isDirectory()) {
                    String packageName = findClassesInDirectory(file, targetClass, "");
                    if (packageName != null) {
                        return packageName;
                    }
                } else if (file.getName().endsWith(".jar")) {
                    JarFile jarFile = null;
                    try {
                        jarFile = new JarFile(file);
                        String packageName = findClassesInJar(jarFile, targetClass);
                        if (packageName != null) {
                            return packageName;
                        }
                    } finally {
                        if (jarFile != null) {
                            jarFile.close();
                        }
                    }
                }
            }
        } catch (IOException e) {
            throw new RuntimeException();
        }

        return null; // 没有找到类
    }

    private static String findClass(String targetClass) throws IOException {
        Enumeration<URL> roots = Thread.currentThread().getContextClassLoader().getResources("");
        while (roots.hasMoreElements()) {
            URL url = roots.nextElement();
            File file = new File(url.getPath());
            if (file.isDirectory()) {
                String className = findClassesInDirectory(file, targetClass, "");
                if (className != null) {
                    return className;
                }
            } else if (file.getName().endsWith(".jar")) {
                JarFile jarFile = new JarFile(file);
                String className = findClassesInJar(jarFile, targetClass);
                jarFile.close();
                if (className != null) {
                    return className;
                }
            }
        }
        return null; // 如果没有找到类，返回null
    }

    private static String findClassesInDirectory(File directory, String targetClass, String packageName) {
        File[] files = directory.listFiles();
        if (files != null) {
            for (File file : files) {
                if (file.isDirectory()) {
                    String className = findClassesInDirectory(file, targetClass, packageName + file.getName() + ".");
                    if (className != null) {
                        return className;
                    }
                } else if (file.getName().endsWith(".class")) {
                    String className = packageName + file.getName().replace(".class", "");
                    if (className.endsWith("." + targetClass)) {
                        return className;
                    }
                }
            }
        }
        return null;
    }

    private static String findClassesInJar(JarFile jarFile, String targetClass) throws IOException {
        Enumeration<JarEntry> entries = jarFile.entries();
        while (entries.hasMoreElements()) {
            JarEntry entry = entries.nextElement();
            if (entry.getName().endsWith(".class")) {
                String className = entry.getName().replace("/", ".").replace(".class", "");
                if (className.endsWith("." + targetClass)) {
                    return className;
                }
            }
        }
        return null;
    }


    // 获取类上的TableComment注解的值
    public static String getClassTableComment(String className) {
        try {
            Class<?> clazz = Class.forName(className);
            if (clazz.isAnnotationPresent(TableComment.class)) {
                TableComment tableComment = clazz.getAnnotation(TableComment.class);
                return tableComment.value();
            }
        } catch (ClassNotFoundException e) {
            throw new RuntimeException();
        }
        return null;
    }

    // 遍历字段，获取字段上的ColumnComment注解
    public static String getFieldComment(String className, String fieldName) {
        try {
            Class<?> clazz = Class.forName(className);
            Field field = clazz.getDeclaredField(fieldName); // 直接获取指定的字段
            if (field.isAnnotationPresent(ColumnComment.class)) {
                ColumnComment columnComment = field.getAnnotation(ColumnComment.class);
                return columnComment.value();  // 返回注解值
            }
        } catch (NoSuchFieldException | ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
        return null;
    }


    public static void main(String[] args) {
        String str = "sys_user_age_di_option";
        int index = str.indexOf('_');
        if (index != -1) {
            String result = str.substring(index + 1);
            System.out.println(result);
        } else {
            System.out.println("Underscore not found in the string.");
        }
    }


}




