package cn.warpin.core.util;


import cn.warpin.core.constant.Constants;
import cn.warpin.core.exception.ResultException;
import cn.warpin.core.result.ResCode;
import org.springframework.http.MediaType;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.web.multipart.MultipartFile;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import static cn.warpin.core.constant.Prop.OssPlatform.LOCAL;
import static cn.warpin.core.constant.Prop.UploadBizType.COMMON;


public class FileUtil {

    public final static String RESOURCE_PATH = Constants.RESOURCE_PATH;

    // 创建文件
    public static String createFile(String filePath, String fileName) {
        createFolder(filePath);
        String symbol = SystemUtil.getSymbol();
        String path = filePath + symbol + fileName;
        File file = new File(path);
        try {
            boolean r = file.createNewFile();
            if (!r) {
                System.out.println("创建文件失败！");
            }
        } catch (IOException e) {
            throw new ResultException(ResCode.FILE_WRITE_FAIL);
        }
        return path;
    }

    // 判断文件是否存在
    public static Boolean checkFileExist(String filePath, String fileName) {
        String symbol = SystemUtil.getSymbol();
        String path = filePath + symbol + fileName;
        File file = new File(path);
        if (file.exists()) {
            return true;
        } else {
            return false;
        }
    }

    // 获取文件类型
    public static String getFileType(String fileName) {
        int index = fileName.lastIndexOf(".");
        String typeName = "";
        if (index < 0) {
            typeName = "png";
        } else {
            typeName = fileName.substring(index + 1);
        }
        return typeName;
    }

    // 重命名文件
    public static String rename(String fileName) {
        Path filePath = Paths.get(fileName);
        String extension = getFileExtension(filePath);
        String uuidStr = UUID.randomUUID().toString();
        String newName = removeFileExtension(filePath) + "_" + uuidStr + extension;
        return newName;
    }

    public static String getFileExtension(Path filePath) {
        String fileName = filePath.getFileName().toString();
        int dotIndex = fileName.lastIndexOf(".");
        if (dotIndex >= 0 && dotIndex < fileName.length() - 1) {
            return fileName.substring(dotIndex);
        }
        return "";
    }

    public static String removeFileExtension(Path filePath) {
        String fileName = filePath.getFileName().toString();
        int dotIndex = fileName.lastIndexOf(".");
        if (dotIndex >= 0) {
            return fileName.substring(0, dotIndex);
        }
        return fileName;
    }

    public static MediaType contentTypeSelector(String extension) {
        MediaType mediaType;
        switch (extension) {
            case ".txt":
                mediaType = MediaType.valueOf(MediaType.TEXT_PLAIN_VALUE);
                break;
            case ".jpg":
            case ".jpeg":
                mediaType = MediaType.valueOf(MediaType.IMAGE_JPEG_VALUE);
                break;
            case ".png":
                mediaType = MediaType.valueOf(MediaType.IMAGE_PNG_VALUE);
                break;
//            case ".zip":
//                mediaType = MediaType.valueOf(MediaType.a);
//                break;
            default:
                mediaType = MediaType.valueOf(MediaType.APPLICATION_OCTET_STREAM_VALUE);
                break;
        }
        return mediaType;
    }


    /**
     * 保存文件到本地
     *
     * @param multipartFile
     * @throws IOException
     * @throws *           /
     * @throws Exception
     */
    public static File multipartFileToFile(MultipartFile multipartFile, String fileName) throws Exception {
        String savePath = RESOURCE_PATH;
        File folder = new File(savePath);
        if (!folder.exists()) {
            folder.mkdirs();
        }
        File file = new File(savePath + fileName);
        multipartFile.transferTo(file);
        return file;
    }

    /**
     * 创建文件夹
     *
     * @param path
     * @throws Exception
     */
    public static void createFolder(String path) {
        File folder = new File(path);
        if (!folder.exists()) {
            folder.mkdirs();
        }
    }

    /**
     * @param file 前端上传的文件
     * @param path 存到本地的路径 含文件名
     * @return 保存后的文件路径(包含文件名)
     * @throws Exception
     */
    public static void saveFile(MultipartFile file, String path) {
//        String fileName = file.getOriginalFilename();
//        String savePath = path + File.separator + fileName;
        File folder = new File(path);
        if (!folder.exists()) {
            folder.mkdirs();
        }
        try {
            file.transferTo(new File(path));
        } catch (IOException e) {
            throw new ResultException(ResCode.FILE_WRITE_FAIL);
        }
    }

    /**
     * 保存文件，文件名已经重命名
     *
     * @param file
     * @param path
     * @param fmtName
     * @return
     */
    public static String saveFile(MultipartFile file, String path, String fmtName) {
        String savePath = path + File.separator + fmtName;
        File folder = new File(path);
        if (!folder.exists()) {
            folder.mkdirs();
        }
        try {
            file.transferTo(new File(savePath));
        } catch (IOException e) {
            throw new ResultException(ResCode.FILE_WRITE_FAIL);
        }
        return savePath;
    }

    public static void deleteFile(String fileName) {
        File file = new File(RESOURCE_PATH + Constants.PS + fileName);
        if (file.exists()) {
            file.delete();
        }
    }

    public static void deleteFile(String path, String fileName) {
        File file = new File(path + fileName);
        if (file.exists()) {
            file.delete();
        }
    }

    public static void saveFile(String base64, String fileName) throws Exception {
        File folder = new File(RESOURCE_PATH);
        if (!folder.exists()) {
            folder.mkdirs();
        }
        try {
            byte[] b = Base64.getDecoder().decode(base64);
            for (int i = 0; i < b.length; ++i) {
                if (b[i] < 0) {
                    b[i] += 256;
                }
            }
            OutputStream out = new FileOutputStream(RESOURCE_PATH + fileName);
            out.write(b);
            out.flush();
            out.close();

        } catch (Exception e) {
            throw e;
        }
    }

    // 将byte数组写入文件
    public static void createFile(String path, byte[] content) throws IOException {
        FileOutputStream fos = new FileOutputStream(path);
        fos.write(content);
        fos.close();
    }

    /**
     * 通过路径获取MultipartFile
     *
     * @param path
     * @return
     * @throws IOException
     */
    public static MultipartFile getMultipartFile(String path) {
        File file = new File(path);
        String fileName = FileUtil.rename(file.getName());
        MultipartFile cMultiFile = null;
        try {
            cMultiFile = new MockMultipartFile("file", fileName, null, new FileInputStream(file));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return cMultiFile;
    }


    /**
     * 查看文件夹下的所有文件
     *
     * @param path
     * @return
     * @throws IOException
     */
    public static List<String> queryFileOfFolder(String path, String suffix) throws IOException {
        List<File> filesInFolder = Files.walk(Paths.get(path)).map(Path::toFile).collect(Collectors.toList());
        List<String> paths = new ArrayList<>();
        for (File f : filesInFolder) {
            String fileName = f.getName();
            String filePath = f.getPath();
            if (StrUtil.isNotEmpty(suffix)) {
                if (!fileName.contains(".")) {
                    continue;
                }
                String fileSuffix = fileName.substring(fileName.indexOf("."));
                if (fileSuffix.equals(suffix)) {
                    paths.add(filePath);
                }
            } else {
                paths.add(filePath);
            }
        }
        return paths;
    }


    public static String saveBufferedImageToFile(String base64string, String extensionName) throws Exception {
        BufferedImage src = base64ToBufferedImage(base64string);
//        BufferedImage src = base64String2BufferedImage(base64string);
        String path = "D:\\resource\\temp\\";
        String fileName = StrUtil.uuid32() + extensionName;
        String pathname = path + fileName;
        File outputfile = new File(pathname);
        ImageIO.write(src, extensionName, outputfile);
        return fileName;
    }

    public static BufferedImage base64ToBufferedImage(String base64String) {
        try {
            byte[] bytes1 = Base64.getDecoder().decode(base64String);
            ByteArrayInputStream bais = new ByteArrayInputStream(bytes1);
            return ImageIO.read(bais);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    private static BufferedImage base64String2BufferedImage(String base64string) throws Exception {
        BufferedImage image = null;
        InputStream stream = BaseToInputStream(base64string);
        image = ImageIO.read(stream);
        return image;
    }

    private static InputStream BaseToInputStream(String base64string) throws Exception {
        ByteArrayInputStream stream = null;

        byte[] bytes1 = Base64.getDecoder().decode(base64string);
        stream = new ByteArrayInputStream(bytes1);

        return stream;
    }

    private static String imageToBase64(BufferedImage bufferedImage) throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ImageIO.write(bufferedImage, "jpg", baos);
        return new String(Base64.getEncoder().encode((baos.toByteArray())));
    }


    /**
     * @param base64        base64编码字符串
     * @param targetPath    图片路径-具体到文件
     * @param extensionName 后缀
     * @return
     * @Description: 将base64编码字符串转换为图片
     * @Author:
     * @CreateTime:
     */
    public static void generateImage(String base64, String targetPath, String extensionName) {
        // 解密
        try {
            // 解密
            java.util.Base64.Decoder decoder = java.util.Base64.getMimeDecoder();
            // 去掉base64前缀 data:image/jpeg;base64,
//            base64 = base64.substring(base64.indexOf(",", 1) + 1, base64.length());
            byte[] b = decoder.decode(base64);
            // 处理数据
            for (int i = 0; i < b.length; ++i) {
                if (b[i] < 0) {
                    b[i] += 256;
                }
            }
            // 保存图片
            String date = DateUtil.longToString(System.currentTimeMillis(), DateUtil.FORMATTYPE_8);

            String filePath = targetPath + "\\" + "al_upload_" + date + "." + extensionName;
            OutputStream out = new FileOutputStream(filePath);
            out.write(b);
            out.flush();
            out.close();
            // 返回图片的相对路径 = 图片分类路径+图片名+图片后缀
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 从url获取Html内容
     *
     * @return
     */
    public static String getHtmlFromUrl(String url) {
        String str = null;
        StringBuffer sb = new StringBuffer();
        try {
            URL uri = new URL(url);
            InputStream in = uri.openStream();
            InputStreamReader isr = new InputStreamReader(in);
            BufferedReader bufr = new BufferedReader(isr);

            while ((str = bufr.readLine()) != null) {
                sb.append(str);
            }
            bufr.close();
            isr.close();
            in.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return sb.toString();
    }

    /**
     * html转PDF, 并上传oss
     *
     * @param htmlText
     * @return
     */
//    public static String htmlTransPDFUploadToOss(String htmlText) {
//        //新建Document对象
//        Document doc = new Document();
//        //添加section
//        Section sec = doc.addSection();
//
//        //添加段落并写入HTML文本
//        sec.addParagraph().appendHTML(htmlText);
//
//        String fileName = StrUtil.uuid32() + ".pdf";
//        // 没有文件夹需要创建文件夹
//        String resource_path = "D:\\resource\\img\\";
//        File folder = new File(resource_path);
//        if (!folder.exists()) {
//            folder.mkdirs();
//        }
//
//        String path = resource_path + fileName;
//        //将文档另存为PDF
//        doc.saveToFile(path, FileFormat.PDF);
//        doc.dispose();
//        MultipartFile mf = getMultipartFile(path);
//        AliOSS.uploadFile(mf, mf.getOriginalFilename(), false);
//        return mf.getOriginalFilename();
//    }


    /**
     * 通过RGB只获取int值
     *
     * @param Red
     * @param Green
     * @param Blue
     * @return
     */
    public static int getIntFromColor(int Red, int Green, int Blue) {
        Red = (Red << 16) & 0x00FF0000; //Shift red 16-bits and mask out other stuff
        Green = (Green << 8) & 0x0000FF00; //Shift Green 8-bits and mask out other stuff
        Blue = Blue & 0x000000FF; //Mask out anything not blue.

        return 0xFF000000 | Red | Green | Blue; //0xFF000000 for 100% Alpha. Bitwise OR everything together.
    }

    public static int getIntFromColor(int Red, int Green, int Blue, int Alpha) {
        Alpha = Alpha << 24;
        Red = (Red << 16) & 0x00FF0000; //Shift red 16-bits and mask out other stuff
        Green = (Green << 8) & 0x0000FF00; //Shift Green 8-bits and mask out other stuff
        Blue = Blue & 0x000000FF; //Mask out anything not blue.

        return 0xFF000000 | Red | Green | Blue | Alpha; //0xFF000000 for 100% Alpha. Bitwise OR everything together.
    }

    /**
     * 获取保存文件的key，包含路径
     *
     * @param filename
     * @param ossContext
     * @param project
     * @param type
     * @param context
     * @param content
     * @param isWindows
     * @return
     */
    public static String getOssKey(String filename, String ossContext, String project, String type, String context, String content, String platform, boolean isWindows) {
        String objectKey;
        String slash = "/";

        // 判断操作系统确定路径分隔符
        if (platform.equals(LOCAL) && isWindows) {
            slash = "\\\\";
        }

        objectKey = project + slash + ossContext;
        if (StrUtil.isEmpty(type)) {
            objectKey = objectKey + slash + COMMON;
        } else {
            objectKey = objectKey + slash + type;
        }

        if (StrUtil.isNotEmpty(context)) {
            objectKey = objectKey + slash + context;
        }

        if (StrUtil.isNotEmpty(content)) {
            objectKey = objectKey + slash + content;
        }

        objectKey = objectKey + slash + filename;
        System.out.println("----- relativePath: " + objectKey);

        return objectKey;
    }


}
