package cn.warpin.core.util;

import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

public class DateUtil {
    public static final String FORMATTYPE_1 = "yyyy-MM-dd HH:mm:ss";
    public static final String FORMATTYPE_2 = "MM月dd日";
    public static final String FORMATTYPE_3 = "yyyyMMdd";
    public static final String FORMATTYPE_4 = "yyyy-MM-dd HH:mm";
    public static final String FORMATTYPE_5 = "yyyy-MM-dd";
    public static final String FORMATTYPE_6 = "HH:mm:ss";
    public static final String FORMATTYPE_7 = "yyyy-MM-dd HH:mm:ss.SSS";
    public static final String FORMATTYPE_8 = "yyyyMMddHHmmss";
    public static final String FORMATTYPE_9 = "yyyyMMddHHmmss.SSS";
    public static final String FORMATTYPE_10 = "yyyyMMddHHmmssSSS";

    /**
     * sql.Date转util.Date
     *
     * @param sqlDate
     * @return
     */
    public static Date sqlDateToUtilDate(Date sqlDate) {
        if (null == sqlDate) {
            return null;
        }
        Date date = new Date(sqlDate.getTime());
        return date;
    }

    /**
     * util.Date转sql.Date
     *
     * @param utilDate
     * @return
     */
    public static java.sql.Date utilDateToSqlDate(Date utilDate) {
        if (null == utilDate) {
            return null;
        }
        java.sql.Date sqlDate = new java.sql.Date(utilDate.getTime());
        return sqlDate;
    }


    // date类型转换为String类型
    // formatType格式为yyyy-MM-dd HH:mm:ss//yyyy年MM月dd日 HH时mm分ss秒
    // data Date类型的时间
    public static String dateToString(Date date, String formatType) {
        return new SimpleDateFormat(formatType).format(date);
    }

    public static String sql_dateToString(java.sql.Date date, String formatType) {
        return new SimpleDateFormat(formatType).format(date);
    }


    // long类型转换为String类型
    // currentTime要转换的long类型的时间
    // formatType要转换的string类型的时间格式
    public static String longToString(long currentTime, String formatType) {
        Date date = null; // long类型转成Date类型
        String strTime = null;
        date = longToDate(currentTime, formatType);
        strTime = dateToString(date, formatType); // date类型转成String
        return strTime;
    }

    // string类型转换为date类型
    // strTime要转换的string类型的时间，formatType要转换的格式yyyy-MM-dd HH:mm:ss//yyyy年MM月dd日
    // HH时mm分ss秒，
    // strTime的时间格式必须要与formatType的时间格式相同
    public static Date stringToDate(String strTime, String formatType) {
        SimpleDateFormat formatter = new SimpleDateFormat(formatType);
        Date date = null;
        try {
            date = formatter.parse(strTime);
        } catch (ParseException e) {
            throw new RuntimeException(e);
        }
        return date;
    }

    // 转成java.sql.date
    public static java.sql.Date stringToSqlDate(String strTime, String formatType) {
        SimpleDateFormat formatter = new SimpleDateFormat(formatType);
        java.sql.Date date = null;
        try {
            date = new java.sql.Date(formatter.parse(strTime).getTime());
        } catch (ParseException e) {
            throw new RuntimeException(e);
        }
        return date;
    }

    // String类型转换为timestamp类型
    public static Timestamp stringToTimestamp(String strTime) {
        return Timestamp.valueOf(strTime);
    }


    // timestamp转String
    public static String timestampToString(Timestamp timestamp, String formatType) {
        return new SimpleDateFormat(formatType).format(timestamp.getTime());
    }

    // long转换为Date类型
    // currentTime要转换的long类型的时间
    // formatType要转换的时间格式yyyy-MM-dd HH:mm:ss//yyyy年MM月dd日 HH时mm分ss秒
    public static Date longToDate(long currentTime, String formatType) {
        Date dateOld = new Date(currentTime); // 根据long类型的毫秒数生命一个date类型的时间
        String sDateTime = dateToString(dateOld, formatType); // 把date类型的时间转换为string
        return stringToDate(sDateTime, formatType);
    }

    // string类型转换为long类型
    // strTime要转换的String类型的时间
    // formatType时间格式
    // strTime的时间格式和formatType的时间格式必须相同
    public static long stringToLong(String strTime, String formatType) {
        Date date = stringToDate(strTime, formatType); // String类型转成date类型
        if (date == null) {
            return 0;
        } else {
            long currentTime = dateToLong(date); // date类型转成long类型
            return currentTime;
        }
    }

    // date类型转换为long类型
    // date要转换的date类型的时间
    public static long dateToLong(Date date) {
        return date.getTime();
    }

    // 获取某天的零时的毫秒数
    // 从当天开始计算，只接受24的倍数，允许正负值，0表示当天
    public static Long getDateZero(int hours) {
        if (hours % 24 != 0) {
            return null;
        }
        long current = System.currentTimeMillis();// 当前时间毫秒数
        long zero = current / (1000 * 3600 * 24) * (1000 * 3600 * 24) - TimeZone.getDefault().getRawOffset();// 今天零点零分零秒的毫秒数
        long timeMillis = zero + hours * 60 * 60 * 1000;
        return timeMillis;
    }

    // 获取一个时间是另一个时间的第几周
    // 起始时间如果是一周的某个时间，那本周只算剩余的天数
    public static int getWeekNum(Date startDate, Date checkDate) {
        int num = 0;
        // 判断起始时间是一周的第几天
        int firstDayOfWeek = dayOfWeek(startDate);
        // 大于7天就算下一周
        int weekMills = 7 * 24 * 3600 * 1000;// 一周的毫秒值
        int leftDay = 7 - firstDayOfWeek;
        long leftMillis = leftDay * 24 * 3600 * 1000;// 剩余天数的毫秒值
        long checkMillis = checkDate.getTime();// 校验时间的毫秒值

        long after1stWeekMills = checkMillis - leftMillis;// 除去第一周剩余的毫秒值
        long minusMills = after1stWeekMills - startDate.getTime();// 递减的毫秒值，从除去第一周的毫秒值开始计算

        while (true) {

            minusMills = minusMills - weekMills;// 减一周的毫秒值
            if (minusMills <= 0) {// 如果减去一周的毫秒值，小于等于0，说明时间已经结束了，跳出循环
                num++;
                break;
            }
            num++;
        }

        return ++num;
    }

    public static int dayOfWeek(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        int w = cal.get(Calendar.DAY_OF_WEEK) - 1;
        if (w < 0) {
            w = 0;
        }
        return w;
    }

    /**
     * 判断比较时间到目标时间是否超过一天
     *
     * @param targetDate 目标时间
     * @param baseDate   基础时间
     * @return 超过返回true，没超过返回false
     */
    public static boolean compareDate(Date targetDate, Date baseDate) {
        // 目标时间和基础时间的毫秒差
        long between = targetDate.getTime() - baseDate.getTime();
        if (between > (24 * 3600000))// 如果毫秒差大于一天，则说明已经超过一天
        {
            return true;
        }
        return false;
    }

    /**
     * 时分秒日期格式转当日时间戳
     *
     * @param time
     * @return
     * @throws ParseException
     */
    public static String timePlusFmt(Date date) {
        String time = DateUtil.dateToString(date, DateUtil.FORMATTYPE_7);
        Date date1 = new Date();
        SimpleDateFormat sdf = new SimpleDateFormat(FORMATTYPE_5);
        String dateStr = sdf.format(date1);
        String fullDateStr = dateStr + " " + time;
        Timestamp timestamp = stringToTimestamp(fullDateStr);
        String result = longToString(timestamp.getTime(), FORMATTYPE_7);
        return result;
    }

    /**
     * 时分秒字符串转当日时间戳
     *
     * @param time
     * @return
     */
    public static String timePlusFmt(String time) {
        String result = null;
        Date date1 = new Date();
        SimpleDateFormat sdf = new SimpleDateFormat(FORMATTYPE_5);
        String dateStr = sdf.format(date1);
        result = dateStr + " " + time;
        return result;
    }

    /**
     * 判断是否是昨天
     *
     * @param oldTime 较小的时间
     * @param newTime 较大的时间 (如果为空 默认当前时间 ,表示和当前时间相比)
     * @return -1 ：同一天. 0：昨天 . 1 ：至少是前天.
     */
    public static boolean isYeaterday(Date oldTime, Date newTime) throws Exception {
        if (null == oldTime) {
            return false;
        }
        if (newTime == null) {
            newTime = new Date();
        }
        SimpleDateFormat format = new SimpleDateFormat(FORMATTYPE_5);
        String todayStr = format.format(newTime);
        Date today = format.parse(todayStr);
        // 昨天 86400000=24*60*60*1000 一天
        if ((today.getTime() - oldTime.getTime()) > 0 && (today.getTime() - oldTime.getTime()) <= 86400000) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 计算两个时间中间的天数
     *
     * @param beginTime
     * @param endTime
     * @return
     * @throws Exception
     */
    public static Integer calTimeRageToDays(Long beginTime, Long endTime) {
        Integer days = 0;
        if (endTime < beginTime) {
            return -1;
        }
        Long time = endTime - beginTime;
        Long daysTime = time / (24 * 3600 * 1000);
        days = daysTime.intValue();
        return days;
    }

    /**
     * 获取日期的年月日
     *
     * @param dateStr      传入的时间，如果不传，则取当前时间
     * @param calendarType Calendar类的常量
     * @return
     */
    public static String getCalendarNum(String dateStr, Integer calendarType) {
        String result = null;
        Date date = null;
        if (StrUtil.isEmpty(dateStr)) {
            date = new Date();
        } else {
            try {
                date = new SimpleDateFormat(FORMATTYPE_5).parse(dateStr);
            } catch (ParseException e) {
                throw new RuntimeException(e);
            }
        }

        Calendar calendar = Calendar.getInstance(Locale.CHINESE);
        calendar.setTime(date);
        // 如果rex为空，则返回年月日，","号隔开
        if (calendarType == null) {
            result = calendar.get(Calendar.YEAR) + "-" + (calendar.get(Calendar.MONTH) + 1) + "-" + calendar.get(Calendar.DAY_OF_MONTH);
            return result;
        }
        switch (calendarType) {
            case Calendar.YEAR:
                result = calendar.get(Calendar.YEAR) + "";
                break;
            case Calendar.MONTH:
                result = (calendar.get(Calendar.MONTH) + 1) + "";
                break;
            case Calendar.DAY_OF_MONTH:
                result = calendar.get(Calendar.DAY_OF_MONTH) + "";
                break;
            case Calendar.DAY_OF_WEEK:
                result = calendar.get(Calendar.DAY_OF_WEEK) + "";
                break;
            case Calendar.DAY_OF_YEAR:
                result = calendar.get(Calendar.DAY_OF_YEAR) + "";
                break;
        }

        return result;
    }

    /**
     * 获取本周第一天日期
     *
     * @return
     */
    public static Timestamp getStartOfWeek() {
        // 获取当前日期
        LocalDate today = LocalDate.now(ZoneId.systemDefault());
        // 计算本周一的日期
        LocalDate startOfWeek = today.with(DayOfWeek.MONDAY);
        // 将LocalDate转换为Timestamp
        return Timestamp.valueOf(startOfWeek.atStartOfDay());
    }


//    public static void main(String[] args) {
//        String r = getCalendarNum("2022-11-23", Calendar.DAY_OF_MONTH);
//        System.out.println(r);
//    }

}
