package cn.warpin.core.security.jwt;

import cn.warpin.core.constant.EnumStore;
import cn.warpin.core.database.redis.RedisOperate;
import cn.warpin.core.util.DateUtil;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * JWT工具类，用于生成和验证JWT令牌。
 */
public class JwtUtils {

    /**
     * 令牌头部键名
     */
    public static final String TOKEN_HEADER = "Authorization";

    /**
     * 令牌头部值前缀
     */
    public static final String TOKEN_PREFIX = "Bearer ";

    /**
     * 主题
     */
    public static final String SUBJECT = "congge";

    /**
     * 令牌有效期
     */
    public static final long EXPIRATION = 1000 * 24 * 60 * 60 * 7;


    /**
     * 密钥
     */
    public static final String APPSECRET_KEY = "congge_secret";

    /**
     * 角色声明键名
     */
    private static final String ROLE_CLAIMS = "rol";

    /**
     * 版本号键名
     */
    private static final String VERSION = "version";

    /**
     * 生成JWT令牌
     *
     * @param loginType 登录类型
     * @param account   账号
     * @param id        ID
     * @return JWT令牌
     */
    public static String createToken(String loginType, String account, String id) {

        Map<String, String> params = getJwtClaimParams(loginType, id);
        String redisKey = params.get("redisKey");
        String token = Jwts
                .builder()
                .setId(id)
                .setSubject(params.get("subject"))
                .setAudience(params.get("audience"))
                .setIssuer(account)
                .claim("account", account)
                .claim("loginType", loginType)
                .claim(VERSION, params.get("version"))
                .setIssuedAt(new Date())
                .setExpiration(new Date(System.currentTimeMillis() + EXPIRATION))
                .signWith(SignatureAlgorithm.HS256, APPSECRET_KEY).compact();

        // 这里要将token保存到redis， 保存token的同时，也保存一个版本号，用来两点登录挤掉线的功能
        Map<String, String> mapValue = new HashMap<>();
        mapValue.put(EnumStore.CACHE_TOKEN_KEY.getValue(), token);
        mapValue.put(EnumStore.CACHE_TOKEN_VERSION_KEY.getValue(), params.get("version"));

        RedisOperate.setMapValue(redisKey, mapValue, 0);

        return token;
    }

    public static Claims checkJWT(String token) {
        try {
            final Claims claims = Jwts.parser().setSigningKey(APPSECRET_KEY).parseClaimsJws(token).getBody();
            return claims;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 获取用户名
     *
     * @param token JWT令牌
     * @return 用户名
     */
    public static String getUsername(String token) {
        Claims claims = Jwts.parser().setSigningKey(APPSECRET_KEY).parseClaimsJws(token).getBody();
        return claims.get("account").toString();
    }

    /**
     * 获取用户角色
     *
     * @param token JWT令牌
     * @return 用户角色
     */
    public static String getUserRole(String token) {
        Claims claims = Jwts.parser().setSigningKey(APPSECRET_KEY).parseClaimsJws(token).getBody();
        return claims.get("rol").toString();
    }

    /**
     * 是否过期
     *
     * @param token JWT令牌
     * @return 如果过期返回true，否则返回false
     */
    public static boolean isExpiration(String token) {
        Claims claims = Jwts.parser().setSigningKey(APPSECRET_KEY).parseClaimsJws(token).getBody();
        return claims.getExpiration().before(new Date());
    }

    /**
     * 获取JWT声明参数
     *
     * @param loginType 登录类型
     * @param id        ID
     * @return JWT声明参数
     */
    public static Map<String, String> getJwtClaimParams(String loginType, String id) {
        Map<String, String> resInfo = new HashMap<>();
        String role = "";
        String redisKey = "";
        String version = "";
        if (loginType.equals(EnumStore.LOGIN_TYPE_ADMIN.getKey())) {
            role = "superAdmin";
            redisKey = EnumStore.CACHE_ADMIN_TOKEN_KEY.getValue() + id;
            version = role + "_" + id + DateUtil.longToString(System.currentTimeMillis(), DateUtil.FORMATTYPE_9);
            resInfo.put("subject", EnumStore.LOGIN_TYPE_ADMIN.getValue());
            resInfo.put("audience", role);
            resInfo.put("redisKey", redisKey);
            resInfo.put("version", version);
        }
        if (loginType.equals(EnumStore.LOGIN_TYPE_EMP.getKey())) {
            role = "employee";
            redisKey = EnumStore.CACHE_EMP_TOKEN_KEY.getValue() + id;
            version = role + "_" + id + DateUtil.longToString(System.currentTimeMillis(), DateUtil.FORMATTYPE_9);
            resInfo.put("subject", EnumStore.LOGIN_TYPE_EMP.getValue());
            resInfo.put("audience", role);
            resInfo.put("redisKey", redisKey);
            resInfo.put("version", version);
        }
        if (loginType.equals(EnumStore.LOGIN_TYPE_USER.getKey())) {
            role = "user";
            redisKey = EnumStore.CACHE_USER_TOKEN_KEY.getValue() + id;
            version = role + "_" + id + DateUtil.longToString(System.currentTimeMillis(), DateUtil.FORMATTYPE_9);
            resInfo.put("subject", EnumStore.LOGIN_TYPE_USER.getValue());
            resInfo.put("audience", role);
            resInfo.put("redisKey", redisKey);
            resInfo.put("version", version);

        }
        return resInfo;
    }


}
