package cn.warpin.core.base.jpa.sql;

import cn.warpin.core.constant.Constants;
import jakarta.annotation.Resource;
import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.stereotype.Component;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;

import static cn.warpin.core.base.jpa.jpaComment.enums.DbTypeEnum.*;

@Component
public class SQLBuilder {

    @Value("${attr.db-schema}")
    private String schema;

    @Resource
    private DataSource dataSource;
    @PersistenceContext
    private EntityManager entityManager;

    private String getDatabaseType() {
        Connection connection = DataSourceUtils.getConnection(dataSource);
        try {
            DatabaseMetaData metaData = connection.getMetaData();
            return metaData.getDatabaseProductName().toUpperCase();
        } catch (SQLException e) {
            throw new RuntimeException("Error accessing database metadata", e);
        } finally {
            DataSourceUtils.releaseConnection(connection, dataSource);
        }
    }

    /**
     * 构建查询列信息的sql
     *
     * @return
     */
    public String buildColumnQuery() {
        String schemaName = Constants.DB_NAME;
        String dbType = getDatabaseType();
        String sql;
        if (dbType.contains(MYSQL.getValue())) {
            sql = "SELECT TABLE_NAME, COLUMN_NAME, COLUMN_COMMENT, DATA_TYPE FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = '" + schemaName + "' AND COLUMN_NAME NOT IN ('created_at','updated_at','created_by','updated_by','deleted_at')";
        } else if (dbType.contains(POSTGRESQL.getValue())) {
            sql = "SELECT table_name, column_name, col_description(format('%I.%I', table_schema, table_name)::regclass, ordinal_position) as column_comment, data_type FROM information_schema.columns WHERE table_schema = 'public' AND column_name NOT IN ('created_at', 'updated_at', 'created_by', 'updated_by', 'deleted_at')";
        } else if (dbType.contains(ORACLE.getValue())) {
            sql = "SELECT table_name, column_name, comments as column_comment, data_type FROM all_col_comments WHERE owner = '" + schemaName.toUpperCase() + "' AND column_name NOT IN ('created_at', 'updated_at', 'created_by', 'updated_by', 'deleted_at')";
        } else if (dbType.contains(SQLSERVER.getValue())) {
            sql = "SELECT t.name AS table_name, c.name AS column_name, ep.value AS column_comment, ty.name AS data_type FROM sys.columns AS c JOIN sys.tables AS t ON c.object_id = t.object_id JOIN sys.types AS ty ON c.user_type_id = ty.user_type_id LEFT JOIN sys.extended_properties AS ep ON ep.major_id = t.object_id AND ep.minor_id = c.column_id AND ep.name = 'MS_Description' WHERE t.name NOT IN ('created_at', 'updated_at', 'created_by', 'updated_by', 'deleted_at')";
        } else {
            throw new IllegalArgumentException("Unsupported database type");
        }
        return sql;
    }

    /**
     * 构建查询表信息的sql
     *
     * @return
     */
    public String buildTableQuery() {
        String schemaName = Constants.DB_NAME;
        String dbType = getDatabaseType();
        String sql;
        if (dbType.contains(MYSQL.getValue())) {
            sql = "SELECT TABLE_NAME, TABLE_COMMENT FROM information_schema.`TABLES` WHERE TABLE_SCHEMA ='" + schemaName + "'";
        } else if (dbType.contains(POSTGRESQL.getValue())) {
            sql = "SELECT table_name, obj_description(to_regclass(table_name)::oid, 'pg_class') as table_comment FROM information_schema.tables WHERE table_schema = 'public'";
        } else if (dbType.contains(ORACLE.getValue())) {
            sql = "SELECT table_name, comments as table_comment FROM all_tab_comments WHERE owner = '" + schemaName.toUpperCase() + "'";
        } else if (dbType.contains(SQLSERVER.getValue())) {
            sql = "SELECT t.name AS table_name, ep.value AS table_comment FROM sys.tables AS t LEFT JOIN sys.extended_properties AS ep ON ep.major_id = t.object_id AND ep.minor_id = 0 AND ep.name = 'MS_Description' WHERE t.schema_id = SCHEMA_ID('" + schemaName + "')";
        } else {
            throw new IllegalArgumentException("Unsupported database type");
        }
        return sql;
    }


    public String buildSpecificColumnQuery(String tableName, String schemaName) {
        String dbType = getDatabaseType(); // 确保已经实现这个方法
        String sql;
        switch (dbType) {
            case "mysql":
                sql = "SELECT COLUMN_NAME, DATA_TYPE, COLUMN_COMMENT, IS_NULLABLE, CHARACTER_MAXIMUM_LENGTH, COLUMN_TYPE, COLUMN_KEY " +
                        "FROM information_schema.COLUMNS " +
                        "WHERE TABLE_NAME = '" + tableName + "' " +
                        "AND TABLE_SCHEMA='" + schemaName + "' " +
                        "AND COLUMN_NAME NOT IN ('id', 'created_at', 'created_by', 'updated_at', 'updated_by', 'unique_tag', 'deleted_at')";
                break;
            case "postgresql":
                sql = "SELECT column_name, data_type, col_description(format('%s.%s', table_schema, table_name)::regclass, ordinal_position) AS column_comment, " +
                        "is_nullable, character_maximum_length, udt_name AS column_type, " +
                        "(CASE WHEN position('p' in column_default) > 0 THEN 'PRI' ELSE '' END) AS column_key " +
                        "FROM information_schema.columns " +
                        "WHERE table_name = '" + tableName + "' " +
                        "AND table_schema = '" + schema + "' " +
                        "AND column_name NOT IN ('id', 'created_at', 'created_by', 'updated_at', 'updated_by', 'unique_tag', 'deleted_at')";
                break;
            case "oracle":
                sql = "SELECT column_name, data_type, data_default AS column_comment, nullable AS is_nullable, data_length AS character_maximum_length, " +
                        "data_type AS column_type, (CASE WHEN constraint_type = 'P' THEN 'PRI' ELSE '' END) AS column_key " +
                        "FROM all_tab_cols LEFT JOIN all_cons_columns USING (column_name, table_name) " +
                        "WHERE table_name = '" + tableName.toUpperCase() + "' " +
                        "AND owner = '" + schemaName.toUpperCase() + "' " +
                        "AND column_name NOT IN ('ID', 'CREATED_AT', 'CREATED_BY', 'UPDATED_AT', 'UPDATED_BY', 'UNIQUE_TAG', 'DELETED_AT')";
                break;
            case "sqlserver":
                sql = "SELECT c.name AS column_name, t.name AS data_type, ep.value AS column_comment, c.is_nullable, c.max_length AS character_maximum_length, " +
                        "t.name AS column_type, (CASE WHEN ik.object_id IS NOT NULL THEN 'PRI' ELSE '' END) AS column_key " +
                        "FROM sys.columns c " +
                        "JOIN sys.types t ON c.user_type_id = t.user_type_id " +
                        "LEFT JOIN sys.extended_properties ep ON ep.major_id = c.object_id AND ep.minor_id = c.column_id AND ep.name = 'MS_Description' " +
                        "LEFT JOIN sys.index_columns ic ON ic.column_id = c.column_id AND ic.object_id = c.object_id " +
                        "LEFT JOIN sys.indexes i ON ic.object_id = i.object_id AND ic.index_id = i.index_id " +
                        "LEFT JOIN sys.key_constraints ik ON i.object_id = ik.parent_object_id AND i.index_id = ik.unique_index_id " +
                        "WHERE c.object_id = OBJECT_ID('" + schemaName + "." + tableName + "') " +
                        "AND c.name NOT IN ('id', 'created_at', 'created_by', 'updated_at', 'updated_by', 'unique_tag', 'deleted_at')";
                break;
            default:
                throw new IllegalArgumentException("Unsupported database type");
        }
        return sql;
    }


}
