package cn.warpin.core.base.dao;

import cn.warpin.core.base.condition.BaseCondition;
import cn.warpin.core.base.condition.QueryCondition;
import cn.warpin.core.base.tableConf.bean.TableConf;
import cn.warpin.core.constant.Prop;
import cn.warpin.core.constant.StandardName;
import cn.warpin.core.exception.ResultException;
import cn.warpin.core.result.ResCode;
import cn.warpin.core.util.ArrayUtil;
import cn.warpin.core.util.ObjectUtil;
import cn.warpin.core.util.StrUtil;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 数据访问工具类，提供了一系列数据库操作的辅助方法。
 */
public class DaoUtil {

    /**
     * 按地理位置范围查找条件的sql
     *
     * @param longitudeGPS 经度
     * @param latitudeGPS  纬度
     * @return 地理位置范围条件的 SQL 语句
     */
    public static String distanceScopeSql(Double longitudeGPS, Double latitudeGPS) {
        return " ROUND(6378.138 * 2 * "
                + "ASIN(SQRT(POW(SIN((" + latitudeGPS + " * PI() / 180 - latitude * PI() / 180) / 2), 2) + "
                + "COS(" + latitudeGPS + "  * PI() / 180) * COS(latitude * PI() / 180) "
                + "* POW(SIN((" + longitudeGPS + " * PI() / 180 - longitude * PI() / 180) / 2),2))) * 1000) ";
    }


    /**
     * 根据查询条件和配置表拼接外部查询条件。
     *
     * @param queryCondition 查询条件
     * @param confs          配置表
     * @param params         参数映射
     * @return 外部查询条件的 SQL 语句
     */
    public static String setExternalCondition(QueryCondition queryCondition, List<TableConf> confs, Map<String, Object> params) {
        StringBuffer conditionSql = new StringBuffer("");
        if (ArrayUtil.isEmpty(confs)) {
            return conditionSql.toString();
        }

        if (queryCondition != null) {
            BaseCondition baseCondition = queryCondition.getBaseCondition();
            Object entity = queryCondition.getEntity();
            try {
                // 先拼接baseCondition的条件，再拼接entity的条件
                if (baseCondition != null) {
                    String ids = baseCondition.getId();

                    // 如果有主键id的只，就需要通过id查询
                    if (StrUtil.isNotEmpty(ids)) {
                        addCondition(conditionSql, Prop.SqlMode.IN, StandardName.ID, ids, params);
                    }
                    // 外键名称和值要跟entity中的字段去匹配，如果匹配到了，就作为条件拼接
                    String foreignKeyName = baseCondition.getForeignKeyName();
                    String foreignKeyValue = baseCondition.getForeignKeyValue();

                    // 这里再按照tableConfig的集合进行参数拼接
                    for (TableConf conf : confs) {
                        String colOrmName = conf.getColumnOrmName();
                        // 如果外键名称与字段名相同，那么就需要用外键查询子表数据
                        if ((StrUtil.isNotEmpty(foreignKeyName) && StrUtil.isNotEmpty(foreignKeyValue)) && foreignKeyName.equals(colOrmName)) {
                            addCondition(conditionSql, Prop.SqlMode.IN, foreignKeyName, foreignKeyValue, params);
                        }
                    }
                }


                // 这里再遍历entity的所有属性，与tableConf进行匹配
                if (entity != null) {
                    Field[] fields = ObjectUtil.getAllFields(entity.getClass());
                    for (int i = 0; i < fields.length; i++) {
                        String name = fields[i].getName(); // 获取属性的名字
                        String upperName = name.substring(0, 1).toUpperCase() + name.substring(1); // 将属性的首字符大写，方便构造get方法
                        Method getMethod = null;// 获取原生类的get方法
                        getMethod = entity.getClass().getMethod("get" + upperName);
                        Object value = getMethod.invoke(entity);
                        String checkVal = value != null ? value + "" : null;
                        if (StrUtil.isNotEmpty(checkVal)) {
                            if (!name.equals(StandardName.ID)) {
                                // 从tableConf中取得sql mode
                                List<TableConf> tcSelected = confs.stream().filter(o -> o.getColumnOrmName().equals(name)).collect(Collectors.toList());
                                TableConf conf = null;
                                if (ArrayUtil.isNotEmpty(tcSelected)) {
                                    conf = tcSelected.get(0);
                                    addCondition(conditionSql, conf.getSqlSpellMode(), name, value, params);
                                }
                            }
                        }
                    }

                }
            } catch (Exception e) {
                throw new ResultException(ResCode.SERVER_COMMON_ERROR);
            }
        }

        return conditionSql.toString();
    }

    /**
     * 根据参数类型添加条件语句。
     *
     * @param sql          SQL 语句
     * @param sqlSpellMode SQL 模式
     * @param propName     属性名称
     * @param value        属性值
     * @param params       参数映射
     */
    private static void addCondition(StringBuffer sql, String sqlSpellMode, String propName, Object value, Map<String, Object> params) {
        if (propName.equals(StandardName.ID) || propName.contains(StandardName.Id)) {
            List<Integer> intVals = StrUtil.ids2IntList(value.toString());
            sql.append(" and ").append(propName).append(" in (:").append(propName).append(")");
            params.put(propName, intVals);
        } else if (sqlSpellMode.equalsIgnoreCase(Prop.SqlMode.EQ)) {
            sql.append(" and ").append(propName).append(" = :").append(propName);
            params.put(propName, value.toString());
        } else if (sqlSpellMode.equalsIgnoreCase(Prop.SqlMode.LIKE)) {
            sql.append(" and ").append(propName).append(" like :").append(propName);
            params.put(propName, value.toString());
        } else if (sqlSpellMode.equalsIgnoreCase(Prop.SqlMode.IN)) {
            String className = value.getClass().getName();
            System.out.println(className);
            switch (className) {
                case "java.lang.String":
                    List<String> StringVals = StrUtil.ids2StringList(value.toString());
                    sql.append(" and ").append(propName).append(" in (:").append(propName).append(")");
                    params.put(propName, StringVals);
                    break;
                case "java.lang.Integer":
                case "java.lang.Double":
                case "java.lang.Float":
                case "java.lang.Long":
                case "java.lang.Short":
                case "java.lang.Byte":
                case "java.math.BigDecimal":
                    List<Integer> intVals = StrUtil.ids2IntList(value.toString());
                    sql.append(" and ").append(propName).append(" in (:").append(propName).append(")");
                    params.put(propName, intVals);
                    break;
                default:
                    break;
            }
        }
    }

}
