package cn.warpin.core.result;

import cn.warpin.core.util.StrUtil;
import lombok.Data;
import org.springframework.data.domain.Page;

import java.util.List;

/**
 * 封装接口统一返回结果。
 */
@Data
public class Result {
    /**
     * 返回的数据
     */
    private Object root;
    /**
     * 返回的消息
     */
    private String msg;
    /**
     * 返回的状态码
     */
    private Integer code;
    /**
     * 返回的数据总数
     */
    private long count;
    /**
     * 返回的总页数
     */
    private int totalPage;
    /**
     * 返回的主键列表
     */
    private List<Integer> primaries;

    /**
     * 构造函数，根据给定的结果码创建一个成功的返回结果。
     *
     * @param root 结果集
     */
    public static Result querySuccess(Object root) {
        return new Result(root, ResCode.SUCCESS);
    }

    /**
     * 请求成功
     *
     * @param root
     * @param totalCount
     * @param totalPage
     * @return
     */
    public static Result querySuccess(Object root, long totalCount, int totalPage) {
        Result rs = new Result(root, ResCode.SUCCESS);
        rs.count = totalCount;
        rs.totalPage = totalPage;
        return rs;
    }

    /**
     * 创建一个成功的返回结果，包含指定的数据和主键列表。
     *
     * @param root      结果集
     * @param primaries 主键列表
     * @return 成功的返回结果
     */
    public static Result querySuccess(Object root, List<Integer> primaries) {
        Result rs = new Result(root, ResCode.SUCCESS);
        rs.primaries = primaries;
        return rs;
    }

    /**
     * 创建一个成功的返回结果，包含指定的数据、总数和总页数，以及主键列表。
     *
     * @param root       结果集
     * @param totalCount 数据总数
     * @param totalPage  总页数
     * @param primaries  主键列表
     * @return 成功的返回结果
     */
    public static Result querySuccess(Object root, long totalCount, int totalPage, List<Integer> primaries) {
        Result rs = new Result(root, ResCode.SUCCESS);
        rs.count = totalCount;
        rs.totalPage = totalPage;
        rs.primaries = primaries;
        return rs;
    }

    /**
     * 创建一个成功的返回结果，不包含数据。
     *
     * @return 成功的返回结果
     */
    public static Result success() {
        return new Result(ResCode.SUCCESS);
    }

    /**
     * 创建一个成功的返回结果，包含指定的状态码和消息。
     *
     * @param rc  状态码
     * @param msg 消息
     * @return 成功的返回结果
     */
    public static Result success(ResCode rc, String msg) {
        return new Result(rc, msg);
    }

    /**
     * 创建一个成功的返回结果，包含指定的数据。
     *
     * @param root 数据
     * @return 成功的返回结果
     */
    public static Result success(Object root) {
        return new Result(root, ResCode.SUCCESS);
    }

    /**
     * 创建一个成功的返回结果，包含指定的数据和状态码。
     *
     * @param root 数据
     * @param rc   状态码
     * @return 成功的返回结果
     */
    public static Result success(Object root, ResCode rc) {
        return new Result(root, rc);
    }

    /**
     * 创建一个成功的返回结果，包含指定的数据、状态码和消息。
     *
     * @param root 数据
     * @param rc   状态码
     * @param msg  消息
     * @return 成功的返回结果
     */
    public static Result success(Object root, ResCode rc, String msg) {
        return new Result(root, rc, msg);
    }

    /**
     * 创建一个成功的返回结果，包含指定的数据和总数。
     *
     * @param root       数据
     * @param totalCount 总数
     * @return 成功的返回结果
     */
    public static Result success(Object root, Long totalCount) {
        return new Result(root, ResCode.SUCCESS, totalCount);
    }

    /**
     * 创建一个成功的返回结果，包含指定的分页数据。
     *
     * @param page 分页数据
     * @return 成功的返回结果
     */
    public static Result success(Page<?> page) {
        return new Result(page, ResCode.SUCCESS);
    }

    /**
     * 创建一个失败的返回结果，不包含消息。
     *
     * @return 失败的返回结果
     */
    public static Result fail() {
        return new Result(ResCode.SERVER_COMMON_ERROR);
    }

    /**
     * 创建一个失败的返回结果，包含指定的状态码。
     *
     * @param rc 状态码
     * @return 失败的返回结果
     */
    public static Result fail(ResCode rc) {
        return new Result(rc);
    }

    /**
     * 创建一个失败的返回结果，包含指定的状态码和消息。
     *
     * @param rc  状态码
     * @param msg 消息
     * @return 失败的返回结果
     */
    public static Result fail(ResCode rc, String msg) {
        return new Result(rc, msg);
    }

    /**
     * 创建一个失败的返回结果，包含指定的数据和状态码。
     *
     * @param root 数据
     * @param rc   状态码
     * @return 失败的返回结果
     */
    public static Result fail(Object root, ResCode rc) {
        return new Result(root, rc);
    }

    /**
     * 创建一个失败的返回结果，包含指定的数据、状态码和消息。
     *
     * @param root 数据
     * @param rc   状态码
     * @param msg  消息
     * @return 失败的返回结果
     */
    public static Result fail(Object root, ResCode rc, String msg) {
        return new Result(root, rc, msg);
    }

    /**
     * 创建一个失败的返回结果，包含指定的状态码和消息。
     *
     * @param resCode 状态码
     * @param msg     消息
     * @return 失败的返回结果
     */
    public static Result fail(Integer resCode, String msg) {
        return new Result(resCode, msg);
    }

    /**
     * 构造函数，根据给定的结果码创建一个成功的返回结果。
     *
     * @param rc 结果码
     */
    public Result(ResCode rc) {
        super();
        this.code = rc.getKey();
        this.msg = rc.getValue();
    }

    /**
     * 构造函数，根据给定的结果码和消息创建一个返回结果。
     *
     * @param rc  结果码
     * @param msg 消息
     */
    public Result(ResCode rc, String msg) {
        super();
        this.code = rc.getKey();
        this.msg = msg;
    }

    /**
     * 创建一个异常返回结果。
     *
     * @param e 异常
     * @return 异常返回结果
     */
    public static Result exception(Exception e) {
        e.printStackTrace();
        return new Result(e, ResCode.SERVER_COMMON_ERROR);
    }

    /**
     * 创建一个自定义状态码和消息的返回结果。
     *
     * @param resCode 自定义状态码
     * @param msg     自定义消息
     */
    public Result(Integer resCode, String msg) {
        super();
        this.code = resCode;
        this.msg = msg;
    }

    /**
     * 构造函数，根据给定的结果码和数据创建一个返回结果。
     *
     * @param root 返回的数据
     * @param rc   结果码
     */
    public Result(Object root, ResCode rc) {
        super();
        this.root = root;
        this.code = rc.getKey();
        this.msg = rc.getValue();
    }

    /**
     * 构造函数，根据给定的结果码、数据和消息创建一个返回结果。
     *
     * @param root 返回的数据
     * @param rc   结果码
     * @param msg  消息
     */
    public Result(Object root, ResCode rc, String msg) {
        super();
        this.root = root;
        this.code = rc.getKey();
        if (StrUtil.isEmpty(msg)) {
            this.msg = rc.getValue();
        } else {
            this.msg = msg;
        }
    }

    /**
     * 构造函数，根据给定的结果码、数据和总数创建一个返回结果。
     *
     * @param root       返回的数据
     * @param rc         结果码
     * @param totalCount 总数
     */
    public Result(Object root, ResCode rc, Long totalCount) {
        super();
        this.root = root;
        this.code = rc.getKey();
        this.msg = rc.getValue();
        this.count = totalCount;
    }


}
