package cn.warpin.core.database.redis;

import io.lettuce.core.RedisClient;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.api.sync.RedisCommands;

import java.util.List;
import java.util.Map;

/**
 * Redis 操作类，封装了一些常用的 Redis 操作方法。
 */
public class RedisOperate {

    /**
     * 保存简单数据
     *
     * @param key        键
     * @param value      值
     * @param expiration 过期时间（秒）
     */
    public static void setValue(String key, String value, long expiration) {
        RedisClient redisClient = RedisClientSingleton.getRedisClient();
        // 获取 StatefulRedisConnection 实例
        StatefulRedisConnection<String, String> connection = redisClient.connect();
        // 获取 RedisCommands 实例
        RedisCommands<String, String> redisCommands = connection.sync();
        redisCommands.set(key, value);
        // 设置列表的过期时间
        if (expiration > 0) {
            redisCommands.expire(key, expiration);
        }
        // 关闭连接
        connection.close();
    }

    /**
     * 获取简单数据
     *
     * @param key 键
     * @return 值
     */
    public static String getValue(String key) {
        RedisClient redisClient = RedisClientSingleton.getRedisClient();
        // 获取 StatefulRedisConnection 实例
        StatefulRedisConnection<String, String> connection = redisClient.connect();
        // 获取 RedisCommands 实例
        RedisCommands<String, String> redisCommands = connection.sync();
        // 使用 GET 命令获取指定的 key 的值
        String value = redisCommands.get(key);
        // 关闭连接
        connection.close();
        // 返回获取的值，如果 key 不存在则返回 null
        return value;
    }

    /**
     * 删除简单数据
     *
     * @param key 键
     */
    public static void deleteValue(String key) {
        RedisClient redisClient = RedisClientSingleton.getRedisClient();
        // 获取 StatefulRedisConnection 实例
        StatefulRedisConnection<String, String> connection = redisClient.connect();
        // 获取 RedisCommands 实例
        RedisCommands<String, String> redisCommands = connection.sync();
        // 使用 DEL 命令删除指定的 key
        redisCommands.del(key);
        // 关闭连接
        connection.close();
    }


    /**
     * 保存map数据
     *
     * @param key        键
     * @param mapValue   Map 数据
     * @param expiration 过期时间（秒）
     */
    public static void setMapValue(String key, Map<String, String> mapValue, long expiration) {
        RedisClient redisClient = RedisClientSingleton.getRedisClient();
        // 获取 StatefulRedisConnection 实例
        StatefulRedisConnection<String, String> connection = redisClient.connect();
        // 获取 RedisCommands 实例
        RedisCommands<String, String> redisCommands = connection.sync();
        redisCommands.hmset(key, mapValue);
        // 获取整个 Map
        // Map<String, String> resultMap = redisCommands.hgetall(key);
        // System.out.println("Table Conf Map: " + resultMap);

        // 设置列表的过期时间
        if (expiration > 0) {
            redisCommands.expire(key, expiration);
        }

        // 关闭连接
        connection.close();

    }

    /**
     * 获取Map数据
     *
     * @param key 键
     * @return Map 数据
     */
    public static Map<String, String> getMapValue(String key) {
        RedisClient redisClient = RedisClientSingleton.getRedisClient();
        // 获取 StatefulRedisConnection 实例
        StatefulRedisConnection<String, String> connection = redisClient.connect();
        // 获取 RedisCommands 实例
        RedisCommands<String, String> redisCommands = connection.sync();
        // 获取整个 Map
        Map<String, String> resultMap = redisCommands.hgetall(key);
        // 关闭连接
        connection.close();
        return resultMap;
    }

    /**
     * 新增数组元素
     *
     * @param key        键
     * @param values     要添加的值列表
     * @param expiration 过期时间（秒）
     */
    public static void setListValue(String key, List<String> values, long expiration) {
        RedisClient redisClient = RedisClientSingleton.getRedisClient();
        StatefulRedisConnection<String, String> connection = redisClient.connect();
        RedisCommands<String, String> redisCommands = connection.sync();
        String[] vs = values.toArray(new String[values.size()]);
        redisCommands.lpush(key, vs);
        // 设置列表的过期时间
        if (expiration > 0) {
            redisCommands.expire(key, expiration);
        }
        connection.close();
    }

    /**
     * 查询列表数据
     *
     * @param key   键
     * @param start 起始位置，默认为0
     * @param stop  结束位置，默认为-1，表示到列表末尾
     * @return 查询到的列表数据
     */
    public static List<String> getListValue(String key, int start, int stop) {
        RedisClient redisClient = RedisClientSingleton.getRedisClient();
        StatefulRedisConnection<String, String> connection = redisClient.connect();
        RedisCommands<String, String> redisCommands = connection.sync();
        List<String> resultList = redisCommands.lrange(key, start, stop);
        connection.close();
        return resultList;
    }


    /**
     * 删除数组元素
     *
     * @param key   键
     * @param value 要删除的值
     */
    public static void deleteListValue(String key, String value) {
        RedisClient redisClient = RedisClientSingleton.getRedisClient();
        StatefulRedisConnection<String, String> connection = redisClient.connect();
        RedisCommands<String, String> redisCommands = connection.sync();
        redisCommands.lrem(key, 0, value);
        connection.close();
    }

    /**
     * 删除数组元素
     *
     * @param key    键
     * @param values 要删除的值列表
     */
    public static void deleteListValue(String key, List<String> values) {
        RedisClient redisClient = RedisClientSingleton.getRedisClient();
        StatefulRedisConnection<String, String> connection = redisClient.connect();
        RedisCommands<String, String> redisCommands = connection.sync();
        for (String value : values) {
            redisCommands.lrem(key, 0, value);
        }
        connection.close();
    }

}
