package cn.warpin.core.base.jpa.jpaComment.repository;

import jakarta.annotation.Resource;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Repository;

import javax.sql.DataSource;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;

import static cn.warpin.core.base.jpa.jpaComment.enums.DbTypeEnum.*;

@Repository
public class JpaCommentRepository {

    @Resource
    public JdbcTemplate jdbcTemplate;

    /**
     * 更新表注释
     *
     * @param tableName
     * @param tableComment
     */
    public void alterTableComment(String tableName, String tableComment) {
        String sql = getTableAlterSql(tableName);
        jdbcTemplate.update(sql, tableComment);
    }

    /**
     * 更新字段注释
     *
     * @param tableName
     * @param columnName
     * @param columnComment
     */
    public void alterColumnComment(String tableName, String columnName, String columnComment) {
        String sql = getColumnCommentAlterSql(tableName, columnName);
        jdbcTemplate.update(sql, columnComment);
    }

    /**
     * 获取更新表注释的方法
     *
     * @param tableName
     * @return
     */
    private String getTableAlterSql(String tableName) {
        String databaseType = getDatabaseType();
        String sql = "";
        if (databaseType.contains(MYSQL.getValue())) {
            sql = "ALTER TABLE " + tableName + " COMMENT = ?";
        } else if (databaseType.contains(SQLSERVER.getValue())) {
            sql = "EXEC sp_updateextendedproperty 'MS_Description', ?, 'SCHEMA', 'dbo', 'TABLE', " + tableName;
        } else if (databaseType.contains(ORACLE.getValue()) || databaseType.contains(POSTGRESQL.getValue())) {
            sql = "COMMENT ON TABLE " + tableName + " IS ?";
        } else {
            throw new IllegalArgumentException("Unsupported database type: " + databaseType);
        }
        return sql;
    }

    /**
     * 获取更新字段注释的方法
     *
     * @param tableName
     * @param columnName
     * @return
     */
    private String getColumnCommentAlterSql(String tableName, String columnName) {
        String databaseType = getDatabaseType();
        String sql = "";
        if (databaseType.contains(MYSQL.getValue())) {
            sql = "ALTER TABLE " + tableName + " MODIFY " + columnName + " ADD COMMENT ?";
        } else if (databaseType.contains(SQLSERVER.getValue())) {
            sql = "EXEC sp_updateextendedproperty 'MS_Description', ?, 'SCHEMA', 'dbo', 'TABLE', " + tableName + ", 'COLUMN', " + columnName;
        } else if (databaseType.contains(ORACLE.getValue()) || databaseType.contains(POSTGRESQL.getValue())) {
            sql = "COMMENT ON COLUMN " + tableName + "." + columnName + " IS ?";
        } else {
            throw new IllegalArgumentException("Unsupported database type: " + databaseType);
        }
        return sql;
    }


    /**
     * 获取数据库类型
     *
     * @return
     * @throws SQLException
     */
    public String getDatabaseType() {
        try {
            DataSource dataSource = jdbcTemplate.getDataSource();
            if (dataSource != null) {
                try (java.sql.Connection connection = dataSource.getConnection()) {
                    DatabaseMetaData metaData = connection.getMetaData();
                    return metaData.getDatabaseProductName().toUpperCase();
                }
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
        return "Unknown";
    }


}
