package cn.wangshuaitong.library.basic.tools;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.Date;
import java.util.function.Function;
import java.util.function.ToLongBiFunction;

/***
 *@author wangt
 *@date 2018/12/29 11:31
 *@describe
 **/
public class DateTimeTools {

    /**
     * 模板代码，只需处理LocalDateTime的计算就可以不需要来回转换
     *
     * @param date
     * @param function
     * @return
     */
    public static Date dataAnalysis4LocalDateTime(Date date, Function<LocalDateTime, LocalDateTime> function) {
        Instant instant = date.toInstant();
        ZoneId zone = ZoneId.systemDefault();
        LocalDateTime localDateTime = LocalDateTime.ofInstant(instant, zone);
        LocalDateTime resultLocalDateTime = function.apply(localDateTime);
        ZonedDateTime zdt = resultLocalDateTime.atZone(zone);
        return Date.from(zdt.toInstant());
    }

    /**
     * 传入时间年初时间
     * 例如：
     * 传入：2009-11-01 11:22:59
     * 返回：2009-01-01 00:00:00
     *
     * @param date
     * @return
     */
    public static Date dateYearEarlyDate(Date date) {
        return dataAnalysis4LocalDateTime(date, localDateTime -> LocalDateTime.of(localDateTime.getYear(), 1, 1, 0, 0, 0));
    }

    /**
     * 传入时间年末时间
     * 例如：
     * 传入：2009-11-01 11:22:59
     * 返回：2009-12-31 23:59:59
     *
     * @param date
     * @return
     */
    public static Date dateYearEndDate(Date date) {
        return dataAnalysis4LocalDateTime(date, localDateTime -> LocalDateTime.of(localDateTime.getYear(), 12, 31, 23, 59, 59));
    }


    /**
     * 传入时间几年前的年初时间
     * 例如：
     * 传入：2009-11-01 11:22:59
     * 返回：2009-01-01 00:00:00
     *
     * @param date
     * @param year
     * @return
     */
    public static Date dateBeforeYearYearEarlyDate(Date date, int year) {
        return dataAnalysis4LocalDateTime(date, localDateTime -> {
            LocalDateTime beforeDateTime = localDateTime.minusYears(year);
            return LocalDateTime.of(beforeDateTime.getYear(), 1, 1, 0, 0, 0);
        });
    }

    /**
     * 传入时间几年后的年末时间
     * 例如：
     * 传入：2009-11-01 11:22:59
     * 返回：2009-01-01 00:00:00
     *
     * @param date
     * @param year
     * @return
     */
    public static Date dateAfterYearYearEndDate(Date date, int year) {
        return dataAnalysis4LocalDateTime(date, localDateTime -> {
            LocalDateTime afterDateTime = localDateTime.plusYears(year);
            return LocalDateTime.of(afterDateTime.getYear(), 12, 31, 23, 59, 59);
        });
    }


    /**
     * 传入时间几月前的月初时间
     * 例如：
     * 传入：2009-11-02 11:22:59
     * 返回：2009-11-01 00:00:00
     *
     * @param date
     * @param month
     * @return
     */
    public static Date dateBeforeMonthMonthEarlyDate(Date date, int month) {
        return dataAnalysis4LocalDateTime(date, localDateTime -> {
            LocalDateTime beforeDateTime = localDateTime.minusMonths(month);
            return LocalDateTime.of(beforeDateTime.getYear(), beforeDateTime.getMonth(), 1, 0, 0, 0);
        });
    }

    /**
     * 传入时间几月前的月初时间
     * 例如：
     * 传入：2009-11-02 11:22:59
     * 返回：2009-11-01 00:00:00
     *
     * @param date
     * @param month
     * @return
     */
    public static Date dateAfterMonthMonthEarlyDate(Date date, int month) {
        return dataAnalysis4LocalDateTime(date, localDateTime -> {
            LocalDateTime beforeDateTime = localDateTime.plusMonths(month);
            return LocalDateTime.of(beforeDateTime.getYear(), beforeDateTime.getMonth(), 1, 0, 0, 0);
        });
    }

    /**
     * 下个月
     *
     * @param date
     * @return
     */
    public static Date nextMonth(Date date) {
        return plusMonth(date, 1);
    }

    /**
     * 月份增加
     * 例如：
     * 传入：2009-11-02 11:22:59
     * 返回：2009-12-02 11:22:59
     *
     * @param date
     * @param increase
     * @return
     */
    public static Date plusMonth(Date date, int increase) {
        return dataAnalysis4LocalDateTime(date, localDateTime -> {
            LocalDateTime beforeDateTime = localDateTime.plusMonths(increase);
            return LocalDateTime.of(beforeDateTime.getYear(), beforeDateTime.getMonth(), beforeDateTime.getDayOfMonth(), beforeDateTime.getHour(), beforeDateTime.getMinute(), beforeDateTime.getSecond());
        });
    }

    /**
     * 下个年
     *
     * @param date
     * @return
     */
    public static Date nextYear(Date date) {
        return plusYear(date, 1);
    }


    /**
     * 年份增加
     * 例如：
     * 传入：2009-11-02 11:22:59
     * 返回：2010-11-02 11:22:59
     *
     * @param date
     * @param increase
     * @return
     */
    public static Date plusYear(Date date, int increase) {
        return dataAnalysis4LocalDateTime(date, localDateTime -> {
            LocalDateTime beforeDateTime = localDateTime.plusYears(increase);
            return LocalDateTime.of(beforeDateTime.getYear(), beforeDateTime.getMonth(), beforeDateTime.getDayOfMonth(), beforeDateTime.getHour(), beforeDateTime.getMinute(), beforeDateTime.getSecond());
        });
    }

    /**
     * 传入时间几月后的月末时间
     * 例如：
     * 传入：2009-11-01 11:22:59
     * 返回：2009-01-01 00:00:00
     *
     * @param date
     * @param month
     * @return
     */
    public static Date dateAfterMonthMonthEndDate(Date date, int month) {
        return dataAnalysis4LocalDateTime(date, localDateTime -> {
            LocalDateTime afterDateTime = localDateTime.plusMonths(month);
            return LocalDateTime.of(afterDateTime.getYear(), afterDateTime.getMonth(), 31, 23, 59, 59);
        });
    }

    /**
     * 传入日期的开始时间
     * 例如：
     * 传入：2009-11-01 11:22:59
     * 返回：2009-11-01 00.00.00
     *
     * @param date
     * @return
     */
    public static Date startTime4SameDay(Date date) {
        return dataAnalysis4LocalDateTime(date, localDateTime -> LocalDateTime.of(localDateTime.getYear(), localDateTime.getMonth(), localDateTime.getDayOfMonth(), 0, 0, 0));
    }

    /**
     * 传入日期的结束时间
     * 例如：
     * 传入：2009-11-01 11:22:59
     * 返回：2009-11-01 23.59.59
     *
     * @param date
     * @return
     */
    public static Date endTime4SameDay(Date date) {
        return dataAnalysis4LocalDateTime(date, localDateTime -> LocalDateTime.of(localDateTime.getYear(), localDateTime.getMonth(), localDateTime.getDayOfMonth(), 23, 59, 59));
    }

    /**
     * 传入日期的月末时间
     * 例如：
     * 传入：2009-11-01 11:22:59
     * 返回：2009-11-31 23.59.59
     *
     * @param date
     * @return
     */
    public static Date endMonthTime4SameDay(Date date) {
        Date nextMonthEarlyDate = dateAfterMonthMonthEarlyDate(date, 1);
        return dataAnalysis4LocalDateTime(nextMonthEarlyDate, localDateTime -> {
            LocalDateTime afterDateTime = localDateTime.minusDays(1);
            return LocalDateTime.of(afterDateTime.getYear(), afterDateTime.getMonth(), afterDateTime.getDayOfMonth(), 23, 59, 59);
        });
    }

    /**
     * 当前时间second秒后的时间
     *
     * @param second
     * @return
     */
    public static Date afterSecondDate(int second) {
        return afterSecondDate(new Date(), second);
    }

    /**
     * 指定时间second秒后的时间
     *
     * @param date
     * @param second
     * @return
     */
    public static Date afterSecondDate(Date date, int second) {
        return timeStampConversion(date, second, (date1, second1) -> date1.getTime() + second1 * 1000);
    }

    /**
     * 表达式基于当前时间和给定数返回一个时间戳的时间
     *
     * @param date
     * @param afterAmount
     * @param operator
     * @param <T>
     * @return
     */
    public static <T> Date timeStampConversion(Date date, T afterAmount, ToLongBiFunction<Date, T> operator) {
        long time = operator.applyAsLong(date, afterAmount);
        return new Date(time);
    }

    /**
     * 当前时间
     *
     * @return
     */
    public static Date nowDate() {
        LocalDateTime localDateTime = LocalDateTime.now();
        ZonedDateTime zdt = localDateTime.atZone(ZoneId.systemDefault());
        return Date.from(zdt.toInstant());
    }

    /**
     * 当前LocalDateTime
     *
     * @return
     */
    public static LocalDateTime now() {
        return LocalDateTime.now();
    }

    /**
     * 传入时间几天之前的日期起始时间
     * 例如：
     * 传入：date -> 2009-11-01 11:22:59, daysAgo -> 1
     * 返回：2009-10-31 00.00.00
     *
     * @param date
     * @param daysAgo
     * @return
     */
    public static Date startTime4SameDayAgo(Date date, Integer daysAgo) {
        return dataAnalysis4LocalDateTime(date, localDateTime -> {
            LocalDateTime daysAgoLocalDateTime = localDateTime.minusDays(daysAgo - 1);
            return LocalDateTime.of(daysAgoLocalDateTime.getYear(), daysAgoLocalDateTime.getMonth(), daysAgoLocalDateTime.getDayOfMonth(), 0, 0, 0);
        });
    }

    /**
     * @param date
     * @param daysLater
     * @return
     */
    public static Date sameDayDaysLater(Date date, Integer daysLater) {
        return dataAnalysis4LocalDateTime(date, localDateTime -> localDateTime.plusDays(daysLater));
    }

    /**
     * @param date
     * @return
     */
    public static Integer getDateYear(Date date) {
        Instant instant = date.toInstant();
        ZoneId zone = ZoneId.systemDefault();
        LocalDateTime localDateTime = LocalDateTime.ofInstant(instant, zone);
        return localDateTime.getYear();
    }

    public static void main(String[] args) {
//        System.out.println(YMDHMS.format(sameDayDaysLater(new Date(), 1)));
//        System.out.println(YMDHMS.format(endTime4SameDay(new Date())));
    }

}
