package cn.wangshuaitong.library.basic.tools;

import java.util.List;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.regex.Pattern;

/***
 *@author wangt
 *@date 2018/8/28 18:10
 *@describe
 **/
public class CommonTools {
    /**
     * 手机号码字符格式
     */
    public static final String PHONE_NUMBER_PATTERN_STR = "^((12[0-9])|(13[0-9])|(15[0-9])|(17[0-9])|(18[0-9]))\\d{8}$";

    /**
     * 手机号码匹配模式对象
     */
    public static final Pattern PHONE_NUMBER_PATTERN = Pattern.compile(PHONE_NUMBER_PATTERN_STR);

    /**
     * 电话号码字符格式
     */
    public static final String TELEPHONE_NUMBER_PATTERN_STR = "^([0-9]{3,4}-?)[0-9]{7,8}$";

    /**
     * 电话号码匹配模式对象
     */
    public static final Pattern TELEPHONE_NUMBER_PATTERN = Pattern.compile(TELEPHONE_NUMBER_PATTERN_STR);

    /**
     * 电话号码字符格式
     */
    public static final String ID_CARD_PATTERN_STR = "^(^[1-9]\\d{7}((0\\d)|(1[0-2]))(([0|1|2]\\d)|3[0-1])\\d{3}$)|(^[1-9]\\d{5}[1-9]\\d{3}((0\\d)|(1[0-2]))(([0|1|2]\\d)|3[0-1])((\\d{4})|\\d{3}[Xx])$)$";

    /**
     * 电话号码匹配模式对象
     */
    public static final Pattern ID_CARD_PATTERN = Pattern.compile(ID_CARD_PATTERN_STR);

    /**
     * 邮箱地址字符格式
     */
    public static final String MAIL_PATTERN_STR = "^([a-zA-Z0-9]*[-_]?[a-zA-Z0-9]+)*@([a-zA-Z0-9]*[-_]?[a-zA-Z0-9]+)+[\\\\.][A-Za-z]{2,3}([\\\\.][A-Za-z]{2})?$";

    /**
     * 邮箱地址匹配模式对象
     */
    public static final Pattern MAIL_PATTERN = Pattern.compile(MAIL_PATTERN_STR);

    /**
     * URL地址字符格式
     */
    public static final String URL_PATTERN_STR = "^(http|https)://[\\w\\.\\-]+(?:/|(?:/[\\w\\.\\-]+)*)?$";

    /**
     * URL地址匹配模式对象
     */
    public static final Pattern URL_PATTERN = Pattern.compile(URL_PATTERN_STR);

    /**
     * URL地址字符格式
     */
    public static final String NUMBER_PATTERN_STR = "^[0-9]*$";

    /**
     * URL地址匹配模式对象
     */
    public static final Pattern NUMBER_PATTERN = Pattern.compile(NUMBER_PATTERN_STR);

    /**
     * 对象是否为空
     *
     * @param value 传入字符串
     * @return
     */
    public static <T> boolean isNull(T value) {
        return null == value;
    }

    /**
     * 对象是否为空
     *
     * @param value 传入字符串
     * @return
     */
    public static <T> boolean isNotNull(T value) {
        return null != value;
    }

    /**
     * 对象是否相等
     *
     * @param value1
     * @param value2
     * @param <T>
     * @return
     */
    public static <T> boolean isEquals(T value1, T value2) {
        if (null == value1 && null == value2) {
            return true;
        }
        if (null == value1) {
            return false;
        }
        return value1.equals(value2);
    }

    /**
     * 对象是否不相等
     *
     * @param value1
     * @param value2
     * @param <T>
     * @return
     */
    public static <T> boolean isNotEquals(T value1, T value2) {
        return !isEquals(value1, value2);
    }

    /**
     * 范围判断
     * 注意：三个参数必须是同一类型
     *
     * @param number 数字
     * @param min    最小值
     * @param max    最大值
     * @param <T>
     * @return
     */
    public static <T extends Comparable> boolean range(T number, T min, T max) {
        return number.compareTo(min) >= 0 && number.compareTo(max) <= 0;
    }

    /**
     * 是否是手机号码
     *
     * @param phoneNumber
     * @return
     */
    public static boolean isPhoneNumber(String phoneNumber) {
        if (StringTools.isEmpty(phoneNumber)) {
            return false;
        }
        return PHONE_NUMBER_PATTERN.matcher(phoneNumber).matches();
    }

    /**
     * 是否不是手机号码
     *
     * @param phoneNumber
     * @return
     */
    public static boolean isNotPhoneNumber(String phoneNumber) {
        return !isPhoneNumber(phoneNumber);
    }

    /**
     * 是否移动电话
     *
     * @param telephoneNumber
     * @return
     */
    public static boolean isTelephoneNumber(String telephoneNumber) {
        if (StringTools.isEmpty(telephoneNumber)) {
            return false;
        }
        return TELEPHONE_NUMBER_PATTERN.matcher(telephoneNumber).matches();
    }

    /**
     * 是否身份证号
     *
     * @param idCard
     * @return
     */
    public static boolean isIdCard(String idCard) {
        if (StringTools.isEmpty(idCard)) {
            return false;
        }
        return ID_CARD_PATTERN.matcher(idCard).matches();
    }

    /**
     * 是否是邮箱地址
     *
     * @param mail
     * @return
     */
    public static boolean isMail(String mail) {
        if (StringTools.isEmpty(mail)) {
            return false;
        }
        return MAIL_PATTERN.matcher(mail).matches();
    }

    /**
     * 是否是URL地址
     *
     * @param url
     * @return
     */
    public static boolean isURL(String url) {
        if (StringTools.isEmpty(url)) {
            return false;
        }
        return URL_PATTERN.matcher(url).matches();
    }

    /**
     * 是否不是URL地址
     *
     * @param url
     * @return
     */
    public static boolean isNotURL(String url) {
        return !isURL(url);
    }

    /**
     * 是否是数字
     *
     * @param number
     * @return
     */
    public static boolean isNumber(String number) {
        if (StringTools.isEmpty(number)) {
            return false;
        }
        return NUMBER_PATTERN.matcher(number).matches();
    }

    /**
     * 字符串不为空时的处理
     *
     * @param value  字符串值
     * @param handle 不为空后的处理器
     */
    public static void noEmptyHandle(String value, Consumer<String> handle) {
        trueThenHandler(StringTools::isNotEmpty, handle, value);
    }

    /**
     * 对象不为空时的处理
     *
     * @param value  对象值
     * @param handle 不为空后的处理器
     */
    public static <T> void noNullHandle(T value, Consumer<T> handle) {
        trueThenHandler(CommonTools::isNotNull, handle, value);
    }

    /**
     * 数组不为空时的处理
     *
     * @param value  对象值
     * @param handle 不为空后的处理器
     */
    public static <T> void noNullHandle4Array(T[] value, Consumer<T[]> handle) {
        trueThenHandler(ArrayTools::isNotEmpty, handle, value);
    }

    /**
     * 集合不为空时的处理
     *
     * @param value  对象值
     * @param handle 不为空后的处理器
     */
    public static <T> void noNullHandle4Collection(List<T> value, Consumer<List<T>> handle) {
        trueThenHandler(CollectionTools::isNotEmpty, handle, value);
    }

    /**
     * 表达式为true时执行一些操作
     *
     * @param judgeExpression 判断表达式
     * @param handle          执行的操作
     * @param value           判断值/执行参数
     * @param <T>             判断值/执行参数类型
     */
    public static <T> void trueThenHandler(Predicate<T> judgeExpression, Consumer<T> handle, T value) {
        trueThenHandler(judgeExpression, value, handle, value);
    }

    /**
     * 表达式为true时执行一些操作
     *
     * @param judgeExpression 判断表达式
     * @param judgeValue      判断值
     * @param handle          执行的操作
     * @param handleValue     执行参数
     * @param <T>             判断值
     * @param <R>             执行参数类型
     */
    public static <T, R> void trueThenHandler(Predicate<T> judgeExpression, T judgeValue, Consumer<R> handle, R handleValue) {
        if (judgeExpression.test(judgeValue)) {
            handle.accept(handleValue);
        }
    }
}
