/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-web-api-contract-js/parameter_type_validator */
var utils = require('vertx-js/util/utils');
var RequestParameter = require('vertx-web-api-contract-js/request_parameter');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JParameterTypeValidator = Java.type('io.vertx.ext.web.api.validation.ParameterTypeValidator');

/**
 Interface for declaration of method for validate a specific parameter type.<br/>
 If you want to implement your own type validator, you need only to implement
 @class
*/
var ParameterTypeValidator = function(j_val) {

  var j_parameterTypeValidator = j_val;
  var that = this;

  /**
   Function that checks if parameter is valid. It returns a RequestParameter object that will be linked inside
   . For more info, check .

   @public
   @param value {string} value of parameter to test 
   @return {RequestParameter} request parameter value
   */
  this.isValid = function(value) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return utils.convReturnVertxGen(RequestParameter, j_parameterTypeValidator["isValid(java.lang.String)"](value));
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Function that checks if array of values of a specific parameter. It returns a RequestParameter object that will
   be linked inside . For more info, check .

   @public
   @param value {Array.<string>} list of values of parameter to test 
   @return {RequestParameter} request parameter value
   */
  this.isValidCollection = function(value) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0] instanceof Array) {
      return utils.convReturnVertxGen(RequestParameter, j_parameterTypeValidator["isValidCollection(java.util.List)"](utils.convParamListBasicOther(value)));
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Returns default value of parameter

   @public

   @return {Object} 
   */
  this.getDefault = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnTypeUnknown(j_parameterTypeValidator["getDefault()"]());
    } else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_parameterTypeValidator;
};

ParameterTypeValidator._jclass = utils.getJavaClass("io.vertx.ext.web.api.validation.ParameterTypeValidator");
ParameterTypeValidator._jtype = {
  accept: function(obj) {
    return ParameterTypeValidator._jclass.isInstance(obj._jdel);
  },
  wrap: function(jdel) {
    var obj = Object.create(ParameterTypeValidator.prototype, {});
    ParameterTypeValidator.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
ParameterTypeValidator._create = function(jdel) {
  var obj = Object.create(ParameterTypeValidator.prototype, {});
  ParameterTypeValidator.apply(obj, arguments);
  return obj;
}
/**
 Create a new type validator for integer values

 @memberof module:vertx-web-api-contract-js/parameter_type_validator
 @param exclusiveMaximum {boolean} If true, value will be mark as valid if value < maximum. If false, value will be mark as valid if value <= maximum. The default value is false. It can be null 
 @param maximum {number} Maximum value. It can be null 
 @param exclusiveMinimum {boolean} If true, value will be mark as valid if value > minimum. If false, value will be mark as valid if value >= minimum. The default value is false. It can be null 
 @param minimum {number} Minimum value. It can be null 
 @param multipleOf {number} Multiple of value. It can be null 
 @param defaultValue {Object} Default value that will be set if parameter is empty or not found. To apply default value you have to set {@link ParameterValidationRule#allowEmptyValue} true. It can be null 
 @return {ParameterTypeValidator} 
 */
ParameterTypeValidator.createIntegerTypeValidator = function() {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] ==='number') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createIntegerTypeValidator(java.lang.Integer)"](utils.convParamInteger(__args[0])));
  }else if (__args.length === 4 && typeof __args[0] ==='number' && typeof __args[1] ==='number' && typeof __args[2] ==='number' && typeof __args[3] ==='number') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createIntegerTypeValidator(java.lang.Double,java.lang.Double,java.lang.Double,java.lang.Integer)"](utils.convParamDouble(__args[0]), utils.convParamDouble(__args[1]), utils.convParamDouble(__args[2]), utils.convParamInteger(__args[3])));
  }else if (__args.length === 6 && typeof __args[0] ==='boolean' && typeof __args[1] ==='number' && typeof __args[2] ==='boolean' && typeof __args[3] ==='number' && typeof __args[4] ==='number' && typeof __args[5] !== 'function') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createIntegerTypeValidator(java.lang.Boolean,java.lang.Double,java.lang.Boolean,java.lang.Double,java.lang.Double,java.lang.Object)"](__args[0], utils.convParamDouble(__args[1]), __args[2], utils.convParamDouble(__args[3]), utils.convParamDouble(__args[4]), utils.convParamTypeUnknown(__args[5])));
  } else throw new TypeError('function invoked with invalid arguments');
};

/**
 Create a new type validator for long integer values

 @memberof module:vertx-web-api-contract-js/parameter_type_validator
 @param exclusiveMaximum {boolean} If true, value will be mark as valid if value < maximum. If false, value will be mark as valid if value <= maximum. The default value is false. It can be null 
 @param maximum {number} Maximum value. It can be null 
 @param exclusiveMinimum {boolean} If true, value will be mark as valid if value > minimum. If false, value will be mark as valid if value >= minimum. The default value is false. It can be null 
 @param minimum {number} Minimum value. It can be null 
 @param multipleOf {number} Multiple of value. It can be null 
 @param defaultValue {Object} Default value that will be set if parameter is empty or not found. To apply default value you have to set {@link ParameterValidationRule#allowEmptyValue} true. It can be null 
 @return {ParameterTypeValidator} 
 */
ParameterTypeValidator.createLongTypeValidator = function() {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] ==='number') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createLongTypeValidator(java.lang.Long)"](utils.convParamLong(__args[0])));
  }else if (__args.length === 4 && typeof __args[0] ==='number' && typeof __args[1] ==='number' && typeof __args[2] ==='number' && typeof __args[3] ==='number') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createLongTypeValidator(java.lang.Double,java.lang.Double,java.lang.Double,java.lang.Long)"](utils.convParamDouble(__args[0]), utils.convParamDouble(__args[1]), utils.convParamDouble(__args[2]), utils.convParamLong(__args[3])));
  }else if (__args.length === 6 && typeof __args[0] ==='boolean' && typeof __args[1] ==='number' && typeof __args[2] ==='boolean' && typeof __args[3] ==='number' && typeof __args[4] ==='number' && typeof __args[5] !== 'function') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createLongTypeValidator(java.lang.Boolean,java.lang.Double,java.lang.Boolean,java.lang.Double,java.lang.Double,java.lang.Object)"](__args[0], utils.convParamDouble(__args[1]), __args[2], utils.convParamDouble(__args[3]), utils.convParamDouble(__args[4]), utils.convParamTypeUnknown(__args[5])));
  } else throw new TypeError('function invoked with invalid arguments');
};

/**
 Create a new type validator for float values

 @memberof module:vertx-web-api-contract-js/parameter_type_validator
 @param exclusiveMaximum {boolean} If true, value will be mark as valid if value < maximum. If false, value will be mark as valid if value <= maximum. The default value is false. It can be null 
 @param maximum {number} Maximum value. It can be null 
 @param exclusiveMinimum {boolean} If true, value will be mark as valid if value > minimum. If false, value will be mark as valid if value >= minimum. The default value is false. It can be null 
 @param minimum {number} Minimum value. It can be null 
 @param multipleOf {number} Multiple of value. It can be null 
 @param defaultValue {Object} Default value that will be set if parameter is empty or not found. To apply default value you have to set {@link ParameterValidationRule#allowEmptyValue} true. It can be null 
 @return {ParameterTypeValidator} 
 */
ParameterTypeValidator.createFloatTypeValidator = function() {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] ==='number') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createFloatTypeValidator(java.lang.Float)"](utils.convParamFloat(__args[0])));
  }else if (__args.length === 4 && typeof __args[0] ==='number' && typeof __args[1] ==='number' && typeof __args[2] ==='number' && typeof __args[3] ==='number') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createFloatTypeValidator(java.lang.Double,java.lang.Double,java.lang.Double,java.lang.Float)"](utils.convParamDouble(__args[0]), utils.convParamDouble(__args[1]), utils.convParamDouble(__args[2]), utils.convParamFloat(__args[3])));
  }else if (__args.length === 6 && typeof __args[0] ==='boolean' && typeof __args[1] ==='number' && typeof __args[2] ==='boolean' && typeof __args[3] ==='number' && typeof __args[4] ==='number' && typeof __args[5] !== 'function') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createFloatTypeValidator(java.lang.Boolean,java.lang.Double,java.lang.Boolean,java.lang.Double,java.lang.Double,java.lang.Object)"](__args[0], utils.convParamDouble(__args[1]), __args[2], utils.convParamDouble(__args[3]), utils.convParamDouble(__args[4]), utils.convParamTypeUnknown(__args[5])));
  } else throw new TypeError('function invoked with invalid arguments');
};

/**
 Create a new type validator for double values

 @memberof module:vertx-web-api-contract-js/parameter_type_validator
 @param exclusiveMaximum {boolean} If true, value will be mark as valid if value < maximum. If false, value will be mark as valid if value <= maximum. The default value is false. It can be null 
 @param maximum {number} Maximum value. It can be null 
 @param exclusiveMinimum {boolean} If true, value will be mark as valid if value > minimum. If false, value will be mark as valid if value >= minimum. The default value is false. It can be null 
 @param minimum {number} Minimum value. It can be null 
 @param multipleOf {number} Multiple of value. It can be null 
 @param defaultValue {Object} Default value that will be set if parameter is empty or not found. To apply default value you have to set {@link ParameterValidationRule#allowEmptyValue} true. It can be null 
 @return {ParameterTypeValidator} 
 */
ParameterTypeValidator.createDoubleTypeValidator = function() {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] ==='number') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createDoubleTypeValidator(java.lang.Double)"](utils.convParamDouble(__args[0])));
  }else if (__args.length === 4 && typeof __args[0] ==='number' && typeof __args[1] ==='number' && typeof __args[2] ==='number' && typeof __args[3] ==='number') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createDoubleTypeValidator(java.lang.Double,java.lang.Double,java.lang.Double,java.lang.Double)"](utils.convParamDouble(__args[0]), utils.convParamDouble(__args[1]), utils.convParamDouble(__args[2]), utils.convParamDouble(__args[3])));
  }else if (__args.length === 6 && typeof __args[0] ==='boolean' && typeof __args[1] ==='number' && typeof __args[2] ==='boolean' && typeof __args[3] ==='number' && typeof __args[4] ==='number' && typeof __args[5] !== 'function') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createDoubleTypeValidator(java.lang.Boolean,java.lang.Double,java.lang.Boolean,java.lang.Double,java.lang.Double,java.lang.Object)"](__args[0], utils.convParamDouble(__args[1]), __args[2], utils.convParamDouble(__args[3]), utils.convParamDouble(__args[4]), utils.convParamTypeUnknown(__args[5])));
  } else throw new TypeError('function invoked with invalid arguments');
};

/**
 Create a new string type validator

 @memberof module:vertx-web-api-contract-js/parameter_type_validator
 @param pattern {string} pattern that string have to match. It can be null 
 @param minLength {number} Minimum length of string. It can be null 
 @param maxLength {number} Maximum length of string. It can be null 
 @param defaultValue {Object} Default value that will be set if parameter is empty or not found. To apply default value you have to set {@link ParameterValidationRule#allowEmptyValue} true. It can be null 
 @return {ParameterTypeValidator} 
 */
ParameterTypeValidator.createStringTypeValidator = function() {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] !== 'function') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createStringTypeValidator(java.lang.Object)"](utils.convParamTypeUnknown(__args[0])));
  }else if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] !== 'function') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createStringTypeValidator(java.lang.String,java.lang.Object)"](__args[0], utils.convParamTypeUnknown(__args[1])));
  }else if (__args.length === 4 && typeof __args[0] === 'string' && typeof __args[1] ==='number' && typeof __args[2] ==='number' && typeof __args[3] !== 'function') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createStringTypeValidator(java.lang.String,java.lang.Integer,java.lang.Integer,java.lang.Object)"](__args[0], utils.convParamInteger(__args[1]), utils.convParamInteger(__args[2]), utils.convParamTypeUnknown(__args[3])));
  } else throw new TypeError('function invoked with invalid arguments');
};

/**
 Create a boolean type validator

 @memberof module:vertx-web-api-contract-js/parameter_type_validator
 @param defaultValue {Object} Default value that will be set if parameter is empty or not found. To apply default value you have to set {@link ParameterValidationRule#allowEmptyValue} true. It can be null 
 @return {ParameterTypeValidator} 
 */
ParameterTypeValidator.createBooleanTypeValidator = function(defaultValue) {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] !== 'function') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createBooleanTypeValidator(java.lang.Object)"](utils.convParamTypeUnknown(defaultValue)));
  } else throw new TypeError('function invoked with invalid arguments');
};

/**
 Create an enum type validator

 @memberof module:vertx-web-api-contract-js/parameter_type_validator
 @param allowedValues {Array.<string>} allowed values. It <b>can't be</b> null 
 @return {ParameterTypeValidator} 
 */
ParameterTypeValidator.createStringEnumTypeValidator = function(allowedValues) {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] === 'object' && __args[0] instanceof Array) {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createStringEnumTypeValidator(java.util.List)"](utils.convParamListBasicOther(allowedValues)));
  } else throw new TypeError('function invoked with invalid arguments');
};

/**
 Create an enum type validator

 @memberof module:vertx-web-api-contract-js/parameter_type_validator
 @param allowedValues {Array.<string>} allowed values. It <b>can't be</b> null 
 @param innerValidator {ParameterTypeValidator} After check if value is one of the lists, you can pass the value to an inner validator. It can be null 
 @return {ParameterTypeValidator} 
 */
ParameterTypeValidator.createEnumTypeValidatorWithInnerValidator = function(allowedValues, innerValidator) {
  var __args = arguments;
  if (__args.length === 2 && typeof __args[0] === 'object' && __args[0] instanceof Array && typeof __args[1] === 'object' && __args[1]._jdel) {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createEnumTypeValidatorWithInnerValidator(java.util.List,io.vertx.ext.web.api.validation.ParameterTypeValidator)"](utils.convParamListBasicOther(allowedValues), innerValidator._jdel));
  } else throw new TypeError('function invoked with invalid arguments');
};

/**
 Create an array type validator

 @memberof module:vertx-web-api-contract-js/parameter_type_validator
 @param arrayMembersValidator {ParameterTypeValidator} Type validator that describe array items. It <b>can't be</b> null 
 @param collectionFormat {string} String that represent a ContainerSerializationStyle. Check out <a href="../../enums.html#ContainerSerializationStyle">ContainerSerializationStyle</a> for more informations. The default value is "csv". It can be null 
 @param maxItems {number} Maximum items in array. It can be null 
 @param minItems {number} Minimum items in array. It can be null 
 @return {ParameterTypeValidator} 
 */
ParameterTypeValidator.createArrayTypeValidator = function() {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createArrayTypeValidator(io.vertx.ext.web.api.validation.ParameterTypeValidator)"](__args[0]._jdel));
  }else if (__args.length === 4 && typeof __args[0] === 'object' && __args[0]._jdel && typeof __args[1] === 'string' && typeof __args[2] ==='number' && typeof __args[3] ==='number') {
    return utils.convReturnVertxGen(ParameterTypeValidator, JParameterTypeValidator["createArrayTypeValidator(io.vertx.ext.web.api.validation.ParameterTypeValidator,java.lang.String,java.lang.Integer,java.lang.Integer)"](__args[0]._jdel, __args[1], utils.convParamInteger(__args[2]), utils.convParamInteger(__args[3])));
  } else throw new TypeError('function invoked with invalid arguments');
};

module.exports = ParameterTypeValidator;