package cn.valot.common.schedule;

import org.springframework.scheduling.Trigger;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.util.*;
import java.util.concurrent.ScheduledFuture;
import java.util.logging.Logger;

/**
 * 定时任务调度服务
 * @author sa@linkot.cn
 */
@Service
public class SchedulerService {
    private final Logger log = Logger.getLogger(SchedulerService.class.getName());
    private final ThreadPoolTaskScheduler threadPoolTaskScheduler;
    private final Map<String, ScheduledFuture<?>> map = new HashMap<>();
    public SchedulerService(ThreadPoolTaskScheduler valotTreadPoolScheduler) {
        this.threadPoolTaskScheduler = valotTreadPoolScheduler;
    }

    /**
     * 当前是否存在该key的任务
     * @param key 任务key
     */
    public boolean exist(String key){
        return map.get(key)!=null;
    }

    /**
     * 不进行管理的一次性任务，任务必定会执行且无法取消
     */
    public void schedule(Runnable runnable, Date startAt) {
        threadPoolTaskScheduler.schedule(runnable, startAt);
    }

    /**
     * 进行管理的一次性任务，任务一段时间后执行，执行前可取消
     */
    public void schedule(String key, Runnable runnable, Date startAt) {
        if (exist(key)) cancel(key, "取消旧任务");
        map.put(key, threadPoolTaskScheduler.schedule(new AutoRemoveTask(key, runnable, this), startAt));
    }

    /**
     * 进行管理的一次性任务，任务一段时间后执行，执行前可取消，执行后不自动移除
     */
    public void scheduleDontRemove(String key, Runnable runnable, Date startAt) {
        if (exist(key)) cancel(key, "取消旧任务");
        map.put(key, threadPoolTaskScheduler.schedule(runnable, startAt));
    }

    /**
     * 进行管理的任务，使用触发器判断是否执行，可取消
     */
    public void schedule(String key, Runnable runnable, Trigger trigger) {
        if (exist(key)) cancel(key, "取消旧任务");
        map.put(key, threadPoolTaskScheduler.schedule(runnable, trigger));
    }

    /**
     * 进行管理的任务，以固定间隔重复执行，可取消
     */
    public void scheduleAtFixedRate(String key, Runnable runnable, Long period) {
        if (exist(key)) cancel(key, "取消旧任务");
        map.put(key, threadPoolTaskScheduler.scheduleAtFixedRate(runnable, period));
    }

    /**
     * 进行管理的任务，指定日期后开始按固定间隔执行，可取消
     */
    public void scheduleAtFixedRate(String key, Runnable runnable, Date startAt, Long period) {
        if (exist(key)) cancel(key, "取消旧任务");
        map.put(key, threadPoolTaskScheduler.scheduleAtFixedRate(runnable, startAt, period));
    }

    /**
     * 取消某个key的定时任务
     */
    public void cancel(String key){
        cancel(key, true, "");
    }

    /**
     * 取消某个key的定时任务
     */
    public void cancel(String key, String reason){
        cancel(key, true, reason);
    }

    /**
     * 取消某个key的定时任务
     * @param interrupt 是否强制取消
     */
    public void cancel(String key, boolean interrupt, String reason){
        if (ObjectUtils.isEmpty(key)) {
            return;
        }
        log.info(String.format("取消定时任务：%s -> %s", key, reason));
        ScheduledFuture<?> s = map.get(key);
        if (s!=null){
            s.cancel(interrupt);
            map.remove(key);
        }else {
            log.info(String.format("任务不在列表中：%s，不存在或已经执行完毕", key));
        }
    }

    /**
     * 获取所有任务 key
     */
    public List<String> getKeys() {
        return new ArrayList<>(map.keySet());
    }


    /**
     * 执行完一次后删除自身的任务
     * @author sa@linkot.cn
     */
    private static class AutoRemoveTask implements Runnable{
        private final String key;
        private final Runnable runnable;
        private final SchedulerService service;

        /**
         * 执行一次后自动删除的任务
         * @param key 任务 key
         * @param runnable 执行
         * @param service 管理定时任务的service
         */
        public AutoRemoveTask(String key, Runnable runnable,
                              SchedulerService service) {
            this.key = key;
            this.runnable = runnable;
            this.service = service;
        }
        @Override
        public void run() {
            runnable.run();
            service.cancel(key);
        }
    }
}
