package cn.valot.common.encrypt;

import cn.hutool.crypto.SmUtil;
import cn.hutool.crypto.asymmetric.SM2;
import cn.hutool.crypto.symmetric.SM4;
import cn.valot.common.data.Bytes;
import cn.valot.common.data.RandomUtil;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.stereotype.Service;

import java.nio.charset.StandardCharsets;

/**
 * SM 国密 加密工具类
 * 需求依赖：cn.hutool:hutool-crypto:5.8.18 、org.bouncycastle:bcprov-jdk15on:1.68
 * @author sa@linkot.cn
 */
@Service
@ConditionalOnClass(name = {"cn.hutool.crypto.SmUtil"})
public class SMService {
    /**
     * 通过id解密
     * @param privateKey 私钥
     * @param raw 密文
     * @return 明文
     */
    public byte[] decryptSM2(String privateKey, byte[] raw){
        String tempKey = SmUtil.sm2().getPublicKeyBase64();
        SM2 sm2 = SmUtil.sm2(privateKey, tempKey);
        return sm2.decrypt(raw);
    }

    public byte[] encryptSM2(String publicKeySM2, byte[] data) {
        String tempKey = SmUtil.sm2().getPrivateKeyBase64();
        SM2 sm2 = SmUtil.sm2(tempKey, publicKeySM2);
        return sm2.encrypt(data);
    }

    public String encryptHexSM2(String publicKeySM2, byte[] data) {
        return Bytes.encodeHex(encryptSM2(publicKeySM2, data));
    }

    /**
     * 随机 16 字节 key
     */
    public String newSM4Key(){
        return RandomUtil.randomAlphanumeric(16);
    }

    /**
     * 通过 key 使用 sm4 加密数据
     * @param key 密钥
     * @param data 数据
     */
    public String encryptSM4(String key, String data){
        SM4 sm4 = SmUtil.sm4(key.getBytes(StandardCharsets.UTF_8));
        return sm4.encryptHex(data);
    }

    /**
     * 通过 key 使用 sm4 解密数据
     * @param key 密钥
     * @param data 数据
     */
    public String decryptSM4(String key, String data){
        SM4 sm4 = SmUtil.sm4(key.getBytes(StandardCharsets.UTF_8));
        return sm4.decryptStr(data);
    }
}
