package cn.valot.common.data.tree;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Tree 工具类
 * @author sa@linkot.cn
 */
public class TreeUtils {
    /**
     * 将集合排列为树形集合
     * @param list 集合对象
     * @param keyFunc 获取 key 的函数，通常为 ::getId()
     * @param pKeyFunc 获取父 key 的函数，通常为 ::getPid()
     * @param rootValue 根节点的父 key
     * @return 排列后的树形集合
     * @param <T> 实现 Tree 接口的类
     * @param <R> Key类型
     * @author sa@linkot.cn
     */
    public static <T extends Tree<T>, R> List<T> marshal(List<T> list, KeyFunc<T, R> keyFunc, KeyFunc<T, R> pKeyFunc, R rootValue){
        return list.stream().filter(e-> rootValue.equals(pKeyFunc.get(e)))
            .peek(e->e.setChildren(marshal(list, keyFunc, pKeyFunc, keyFunc.get(e))))
            .collect(Collectors.toList());
    }

    /**
     * 将树形结构展开为 list
     * @param tree 树形 list
     * @return list
     */
    public static <T extends Tree<T>, R> List<T> unmarshal(List<T> tree){
        List<T> result = new ArrayList<>();
        expand(tree, result);
        return result;
    }
    /**
     * 将树形节点展开为 list，并添加进 result
     * @param nodes 树形 list
     * @param result 结果集
     */
    public static <T extends Tree<T>, R> void expand(List<T> nodes, List<T> result){
        if (nodes != null && !nodes.isEmpty()){
            for (T node : nodes) {
                result.add(node);
                expand(node.getChildren(), result);
            }
        }
    }

    public interface KeyFunc<T, R>{
        R get(T e);
    }
}
