package cn.valot.common.data;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.Date;

@Service
@ConditionalOnClass(name = {"org.springframework.data.redis.core.RedisTemplate", "com.alibaba.fastjson.JSON"})
public class RedisService {
    public RedisService(RedisTemplate<String, Object> valotRedisTemplate) {
        this.redisTemplate = valotRedisTemplate;
    }

    private final RedisTemplate<String, Object> redisTemplate;

    public void set(String key, Object value) {
        redisTemplate.opsForValue().set(key, value);
    }

    public void set(String key, Object value, Date expireAt) {
        set(key, value);
        expireAt(key, expireAt);
    }

    /**
     * 键值对、period 后过期
     *
     * @param key    key
     * @param value  val
     * @param period 时间、单位为毫秒
     */
    public void set(String key, Object value, Long period) {
        set(key, value, new Date(System.currentTimeMillis() + period));
    }

    /**
     * 参考 redis setnx
     */
    public Boolean setIfNotExist(String key, Object data) {
        return redisTemplate.opsForValue().setIfAbsent(key, data);
    }

    /**
     * 参考 redis setnx, expireAt 为过期时间
     */
    public Boolean setIfNotExist(String key, Object data, Date expireAt) {
        Boolean result = setIfNotExist(key, data);
        if (result) {
            expireAt(key, expireAt);
        }
        return result;
    }

    public void setHash(String key, String field, Object value) {
        redisTemplate.opsForHash().put(key, field, value);
    }

    /**
     * 设置 key 的过期时间
     *
     * @param key  key
     * @param date 过期时间
     */
    public void expireAt(String key, Date date) {
        redisTemplate.expireAt(key, date);
    }

    public Object get(String key) {
        return redisTemplate.opsForValue().get(key);
    }

    public <T> T get(String key, Class<T> clz) {
        Object o = get(key);
        if (o instanceof JSONObject) {
            JSONObject data = (JSONObject) o;
            return data.toJavaObject(clz);
        }
        return (T) o;
    }

    public Object get(String key, Object field) {
        return redisTemplate.opsForHash().get(key, field);
    }

    public <T> T get(String key, String field, Class<T> clz) {
        Object o = get(key, field);
        if (o instanceof JSONObject) {
            JSONObject data = (JSONObject) o;
            return data.toJavaObject(clz);
        }
        return (T) o;
    }

    public void delete(String key) {
        redisTemplate.delete(key);
    }

    public void deleteHash(String key, Object... field){
        redisTemplate.opsForHash().delete(key, field);
    }


    /**
     * key 数值增加 delta
     * @param key key
     * @param delta 增加数
     */
    public Long incr(String key, Long delta) {
        return redisTemplate.opsForValue().increment(key, delta);
    }

    /**
     * 获取后删除key
     * @param key key
     */
    public Object getThenDelete(String key) {
        Object res = get(key);
        delete(key);
        return res;
    }
}
