package cn.valot.common.data;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.function.BiPredicate;
import java.util.function.Function;

/**
 * Map工具类
 * @author sa@linkot.cn
 */
public final class Maps{
    /**
     * 把所有 dict 转化为 hashmap
     * @param dicts 键值对集
     * @return hashmap
     */
    @SafeVarargs
    public static <K, V> HashMap<K, V> asMap(Dict<K, V>... dicts){
        HashMap<K, V> map = new HashMap<>();
        putAll(map, dicts);
        return map;
    }
    /**
     * 把所有 dict 转化为 hashtable
     * @param dicts 键值对集
     * @return hashtable
     */
    @SafeVarargs
    public static <K, V> Hashtable<K, V> asTable(Dict<K, V>... dicts){
        Hashtable<K, V> map = new Hashtable<>();
        putAll(map, dicts);
        return map;
    }


    /**
     * 往 map 中追加键值对
     * @param map 原map
     * @param dicts 键值对集
     */
    @SafeVarargs
    private static <K, V> void putAll(Map<K, V> map, Dict<K, V>... dicts) {
        for (Dict<K, V> dict : dicts) {
            map.put(dict.getKey(), dict.getValue());
        }
    }


    /**
     * 把 list 转化为 map
     * @param list 所有元素
     * @param keyFunc 转化为 key 的方法
     * @return key 为通过 keyFunc 转换的， value 为原对象
     */
    public static <K, V> Map<K, V> parseMap(List<V> list, Function<V, K> keyFunc){
        HashMap<K, V> map = new HashMap<>();
        for (V v : list) {
            map.put(keyFunc.apply(v), v);
        }
        return map;
    }
    /**
     * 把 list 转化为 map
     * @param list 所有元素
     * @param keyFunc 转化为 key 的方法
     * @param valueFunc 转化为 value 的方法
     * @return 通过 keyFunc 和 valueFunc 映射的新 map
     */
    public static <K, V, R> Map<K, R> parseMap(List<V> list, Function<V, K> keyFunc, Function<V, R> valueFunc){
        HashMap<K, R> map = new HashMap<>();
        for (V v : list) {
            map.put(keyFunc.apply(v), valueFunc.apply(v));
        }
        return map;
    }

    /**
     * 过滤 map 中的键值对
     * @param map 原 map
     * @param func 过滤函数
     * @return 条件为 true 的map
     */
    public static <K, V> Map<K, V> filter(Map<K, V> map, BiPredicate<K, V> func){
        Map<K, V> m = new HashMap<>();
        for (K k : map.keySet()) {
            if (func.test(k, map.get(k))){
                m.put(k, map.get(k));
            }
        }
        return m;
    }
}

