package cn.valot.common.data;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

/**
 * 日期范围工具类
 * @author sa@linkot.cn
 */
public final class Calendars {
    private static final DateTimeFormatter DATE_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final DateTimeFormatter DATE_FORMAT_FLAT = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");

    public static String currentTimeString(){
        return DATE_FORMAT_FLAT.format(LocalDateTime.now());
    }

    /**
     * 本月时间域
     */
    public static String[] getMonthArea(){
        int year = getYear();
        int month = getMonth();
        Area area = getArea(year, month);
        return concatYear(year, area.start, area.end);
    }

    public static String[] concatYear(int year, String area, String area1) {
        return new String[]{year + "-" + area, year + "-" + area1};
    }
    public static String[] concatYear(int year, String area, int year2, String area1) {
        return new String[]{year + "-" + area, year2 + "-" + area1};
    }

    public static LocalDate parseDate(String str){
        try{
            return LocalDate.from(DATE_FORMAT.parse(str));
        }catch (Exception ex){
            throw new RuntimeException(ex);
        }
    }
    public static LocalDate parseFlatDate(String str){
        try{
            return LocalDate.from(DATE_FORMAT.parse(str));
        }catch (Exception ex){
            throw new RuntimeException(ex);
        }
    }

    /**
     * 上个月的时间域
     */
    public static String[] getPreventMonthArea(){
        Calendar i = Calendar.getInstance(Locale.SIMPLIFIED_CHINESE);
        int year = i.get(Calendar.YEAR);
        int month = i.get(Calendar.MONTH);
        if (month == 0){
            month = 11;
            year--;
        }else {
            month--;
        }
        Area area = getArea(year, month);
        return concatYear(year, area.start, area.end);
    }
    /**
     * 例：["2023-01-01 00:00:00","2023-01-01 23:59:59"]
     */
    public static String[] today(){
        String date = DATE_FORMAT.format(LocalDate.now());
        return new String[]{date + " 00:00:00", date + " 23:59:59"};
    }

    public static Area getArea(int year, int month){
        if (month == -1){
            month = 11;
        }
        if (month == 12){
            month = 0;
        }
        if (year %4 == 0 && month == 2){
            return Area.FEB_LARGE;
        }
        return Area.values()[month];
    }
    /**
     * 例：["2023-01-01","2023-12-31"]
     */
    public static String[] thisYear() {
        int year = getYear();
        return concatYear(year, "01-01", "12-31");
    }

    public static int getYear() {
        return Calendar.getInstance().get(Calendar.YEAR);
    }
    private static int getMonth() {
        return Calendar.getInstance().get(Calendar.MONTH);
    }

    /**
     * 例：["2022-04-01","2023-03-31"]
     */
    public static String[] getYearArea(){
        int year = getYear();
        int month = getMonth();
        Area end = getArea(year, month);
        Area start = getArea(year - 1, month + 1);
        return concatYear(year -1, start.start, year, end.end);
    }

    public enum Area {
        JAN("01-01", "01-31"),
        FEB("02-01", "02-28"),
        MAR("03-01", "03-31"),
        APR("04-01", "04-30"),
        MAY("05-01", "05-31"),
        JUN("06-01", "06-30"),
        JUL("07-01", "07-31"),
        AUG("08-01", "08-31"),
        SEP("09-01", "09-30"),
        OCT("10-01", "10-31"),
        NOV("11-01", "11-30"),
        DEC("12-01", "12-31"),
        FEB_LARGE("02-01", "02-29");

        public final String start;
        public final String end;
        Area(String start, String end) {
            this.start = start;
            this.end = end;
        }
    }

    /**
     * 以当前时间为基本进行偏移（加减）
     * @param field 日期单位
     * @param period 数量（可以为负数）
     */
    public static Date add(int field, int period){
        Calendar ins = Calendar.getInstance();
        ins.add(field, period);
        return ins.getTime();
    }
}