package cn.tworice.common.util.excel;


import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.List;

/**
 * 数据导入
 * @param <E> 与数据表映射的表列名
 */
@Slf4j
public class InductsListener<E> extends AnalysisEventListener<E> {
    /**
     * 每隔20条存储数据库，实际使用中可以3000条，然后清理list ，方便内存回收
     */
    private static final int BATCH_COUNT = 20;
    List<E> list = new ArrayList<>();

    /**
     * 假设这个是一个DAO，当然有业务逻辑这个也可以是一个service。当然如果不用存储这个对象没用。
     */
    private final InductsListenerService<E> service;

    /**
     * 每次创建Listener的时候需要把spring管理的类传进来
     * @param service 实现了InductsListenerService接口的业务层
     */
    public InductsListener(InductsListenerService<E> service) {
        this.service = service;
    }

    /**
     * 这个每一条数据解析都会来调用
     * @param data 行数据
     */
    @Override
    public void invoke(E data, AnalysisContext context) {
        log.info("解析到一条数据:{}", JSON.toJSONString(data));
        list.add(data);
        // 达到BATCH_COUNT了，需要去存储一次数据库，防止数据几万条数据在内存，容易OOM
        if (list.size() >= BATCH_COUNT) {
            this.saveData();
            list.clear();
        }
    }

    /**
     * 所有数据解析完成后调用
     * 这里也要保存数据，确保最后遗留的数据也存储到数据库
     */
    @Override
    public void doAfterAllAnalysed(AnalysisContext context) {
        this.saveData();
    }

    /**
     * 存储数据库
     */
    private void saveData() {
        this.service.saveBatch(list);
    }
}
