package cn.tworice.common.util;

import com.google.zxing.*;
import com.google.zxing.client.j2se.BufferedImageLuminanceSource;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.common.HybridBinarizer;
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel;
import lombok.Setter;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.geom.RoundRectangle2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.OutputStream;
import java.util.Hashtable;
import java.util.Random;

/**
 * 通过设置前景色、背景色、二维码尺寸、LOGO图片、背景图等生成二维码
 * @author 二饭
 */
public class QRCodeUtil {
    /**
     * 编码格式
     */
    private final String CHARSET = "utf-8";
    /**
     * 图片格式
     */
    private final String FORMAT_NAME = "JPG";
    /**
     * 二维码尺寸
     *  设置二维码的尺寸
     */
    @Setter
    private int QRCODE_SIZE = 300;
    /**
     * LOGO宽度
     */
    private final int WIDTH = 60;
    /**
     * LOGO高度
     */
    private final int HEIGHT = 60;
    /**
     * 前景色
     */
    private int COLOR=-16777216;
    /**
     * 背景色
     */
    private int BGC=0xFFFFFFFF;
    /**
     * 二维码白边边缘 可设置范围为0-10，但仅四个变化0 1(2) 3(4 5 6) 7(8 9 10)
     * -- SETTER --
     *  设置二维码边缘
     *

     */
    @Setter
    private int MARGIN=1;
    /**
     * 纠错等级
     * -- SETTER --
     *  设置二维码的纠错等级
     *

     */
    @Setter
    private ErrorCorrectionLevel Level=ErrorCorrectionLevel.H;

    /**
     * @param color 前景色，如：#2C81FF 格式的字符串
     * @author 二饭
     * @email 1786705138@qq.com
     */
    public void setColor(String color) {
        this.COLOR=Color.decode(color).getRGB();
    }

    /**
     * @param bgc 背景色，如：#2C81FF 格式的字符串
     * @author 二饭
     */
    public void setBgc(String bgc) {
        this.BGC=Color.decode(bgc).getRGB();
    }

    /**
     * @param content 内容
     * @param imgPath LOGO图片
     * @param destPath 存储路径
     * @param needCompress 是否压缩
     * @author 二饭
     */
    public String encodeTest(String content, String imgPath, String destPath,boolean needCompress)throws Exception{
        Hashtable<EncodeHintType, Object> hints = new Hashtable<EncodeHintType, Object>();
        hints.put(EncodeHintType.ERROR_CORRECTION, this.Level);// 纠错等级，越高存储越多，越不易被识别
        hints.put(EncodeHintType.CHARACTER_SET, CHARSET);// 编码字符集
        hints.put(EncodeHintType.MARGIN, this.MARGIN);// 设置白边边缘

        // 创建位矩阵对象 内容、图片类型、宽度、高度、
        BitMatrix bitMatrix = new MultiFormatWriter().encode(content,BarcodeFormat.QR_CODE, QRCODE_SIZE, QRCODE_SIZE, hints);

        // 获取宽度高度
        int width = bitMatrix.getWidth();
        int height = bitMatrix.getHeight();

        // 生成图像
        BufferedImage image = new BufferedImage(width, height,BufferedImage.TYPE_INT_RGB);
        for (int x = 0; x < width; x++) {
            for (int y = 0; y < height; y++) {
                image.setRGB(x, y, bitMatrix.get(x, y) ? this.COLOR
                        : this.BGC);
            }
        }

        mkdirs(destPath);
        String file = System.currentTimeMillis()+".jpg";
        ImageIO.write(image, FORMAT_NAME, new File(destPath+"/"+file));
        return file;
    }

    /**
     * @param content 内容
     * @param imgPath 路径
     * @param needCompress 是否需要压缩
     * @author 二饭
     */
    public BufferedImage createImage(String content, String imgPath,boolean needCompress) throws Exception {
        Hashtable<EncodeHintType, Object> hints = new Hashtable<EncodeHintType, Object>();
        hints.put(EncodeHintType.ERROR_CORRECTION, this.Level);// 纠错等级，越高存储越多，越不易被识别
        hints.put(EncodeHintType.CHARACTER_SET, CHARSET);// 编码字符集
        hints.put(EncodeHintType.MARGIN, this.MARGIN);// 设置白边边缘

        // 创建位矩阵对象 内容、图片类型、宽度、高度、
        BitMatrix bitMatrix = new MultiFormatWriter().encode(content,BarcodeFormat.QR_CODE, QRCODE_SIZE, QRCODE_SIZE, hints);

        // 获取宽度高度
        int width = bitMatrix.getWidth();
        int height = bitMatrix.getHeight();

        // 生成图像
        BufferedImage image = new BufferedImage(width, height,BufferedImage.TYPE_INT_RGB);
        for (int x = 0; x < width; x++) {
            for (int y = 0; y < height; y++) {
                image.setRGB(x, y, bitMatrix.get(x, y) ? this.COLOR
                        : this.BGC);
            }
        }

        if (imgPath == null || "".equals(imgPath)) {
            return image;
        }
        // 插入图片
        insertImage(image, imgPath, needCompress);
        return image;
    }

    /**
     * 插入LOGO
     *
     * @param source 二维码图片
     * @param imgPath LOGO图片地址
     * @param needCompress 是否压缩
     */
    private void insertImage(BufferedImage source, String imgPath,boolean needCompress) throws Exception {
        File file = new File(imgPath);
        if (!file.exists()) {
            return;
        }
        Image src = ImageIO.read(new File(imgPath));
        int width = src.getWidth(null);
        int height = src.getHeight(null);
        if (needCompress) { // 压缩LOGO
            if (width > WIDTH) {
                width = WIDTH;
            }
            if (height > HEIGHT) {
                height = HEIGHT;
            }
            Image image = src.getScaledInstance(width, height,
                    Image.SCALE_SMOOTH);
            BufferedImage tag = new BufferedImage(width, height,
                    BufferedImage.TYPE_INT_RGB);
            Graphics g = tag.getGraphics();
            g.drawImage(image, 0, 0, null); // 绘制缩小后的图
            g.dispose();
            src = image;
        }
        // 插入LOGO
        Graphics2D graph = source.createGraphics();
        int x = (QRCODE_SIZE - width) / 2;
        int y = (QRCODE_SIZE - height) / 2;
        graph.drawImage(src, x, y, width, height, null);
        Shape shape = new RoundRectangle2D.Float(x, y, width, width, 6, 6);
        graph.setStroke(new BasicStroke(3f));
        graph.draw(shape);
        graph.dispose();
    }

    /**
     * 生成二维码(内嵌LOGO)
     *
     * @param content 内容
     * @param imgPath LOGO地址
     * @param destPath 存放目录
     * @param needCompress 是否压缩LOGO
     * @throws Exception 抛出异常
     */
    public String encode(String content, String imgPath, String destPath,boolean needCompress) throws Exception {
        BufferedImage image = createImage(content, imgPath,needCompress);
        mkdirs(destPath);
        String file = new Random().nextInt(99999999)+".jpg";
        ImageIO.write(image, FORMAT_NAME, new File(destPath+"/"+file));
        return file;
    }

    /**
     * 当文件夹不存在时，mkdirs会自动创建多层目录，区别于mkdir．(mkdir如果父目录不存在则会抛出异常)
     * @param destPath 存放目录
     */
    public void mkdirs(String destPath) {
        File file =new File(destPath);
        //当文件夹不存在时，mkdirs会自动创建多层目录，区别于mkdir．(mkdir如果父目录不存在则会抛出异常)
        if (!file.exists() && !file.isDirectory()) {
            file.mkdirs();
        }
    }

    /**
     * 生成二维码(内嵌LOGO)
     *
     * @param content 内容
     * @param imgPath LOGO地址
     * @param destPath 存储地址
     */
    public void encode(String content, String imgPath, String destPath)
            throws Exception {
        encode(content, imgPath, destPath, false);
    }

    /**
     * 生成二维码
     *
     * @param content 内容
     * @param destPath 存储地址
     * @param needCompress 是否压缩LOGO
     */
    public void encode(String content, String destPath,boolean needCompress) throws Exception {
        encode(content, null, destPath, needCompress);
    }

    /**
     * 生成二维码
     *
     * @param content 内容
     * @param destPath 存储地址
     */
    public void encode(String content, String destPath) throws Exception {
        encode(content, null, destPath, false);
    }

    /**
     * 生成二维码(内嵌LOGO)
     *
     * @param content 内容
     * @param imgPath LOGO地址
     * @param output 输出流
     * @param needCompress 是否压缩LOGO
     */
    public void encode(String content, String imgPath,OutputStream output, boolean needCompress) throws Exception {
        BufferedImage image = createImage(content, imgPath,needCompress);
        ImageIO.write(image, FORMAT_NAME, output);
    }

    /**
     * 生成二维码
     *
     * @param content 内容
     * @param output 输出流
     */
    public void encode(String content, OutputStream output)throws Exception {
        encode(content, null, output, false);
    }

    /**
     * 解析二维码
     *
     * @param file 二维码图片
     * @return 结果
     * @throws Exception 抛出异常
     */
    public String decode(File file) throws Exception {
        BufferedImage image;
        image = ImageIO.read(file);
        if (image == null) {
            return null;
        }
        BufferedImageLuminanceSource source = new BufferedImageLuminanceSource(image);
        BinaryBitmap bitmap = new BinaryBitmap(new HybridBinarizer(source));
        Result result;
        Hashtable<DecodeHintType, Object> hints = new Hashtable<DecodeHintType, Object>();
        hints.put(DecodeHintType.CHARACTER_SET, CHARSET);
        result = new MultiFormatReader().decode(bitmap, hints);
        String resultStr = result.getText();
        return resultStr;
    }

    /**
     * 解析二维码
     *
     * @param path 二维码图片地址
     */
    public String decode(String path) throws Exception {
        return decode(new File(path));
    }


}
