package cn.twelvet.oss.http;

import cn.twelvet.oss.service.OssTemplate;
import com.amazonaws.services.s3.model.Bucket;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import org.springframework.http.HttpStatus;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * oss 对外提供服务端点
 *
 * @author twelvet
 */
@Validated
@RestController
@RequestMapping("${oss.http.prefix:}/oss")
public class OssEndpoint {

	/**
	 * OSS操作模板
	 */
	private final OssTemplate ossTemplate;

	/**
	 * 注入OssTemplate
	 * @param ossTemplate OssTemplate
	 */
	public OssEndpoint(OssTemplate ossTemplate) {
		this.ossTemplate = ossTemplate;
	}

	/**
	 * 创建桶
	 * @param bucketName 桶名称
	 * @return Bucket
	 */
	@PostMapping("/bucket/{bucketName}")
	public Bucket createBucket(@PathVariable @NotBlank String bucketName) {
		ossTemplate.createBucket(bucketName);
		return ossTemplate.getBucket(bucketName).get();
	}

	/**
	 * 获取所有桶名称
	 * @return List Bucket
	 */
	@GetMapping("/bucket")
	public List<Bucket> getBuckets() {
		return ossTemplate.getAllBuckets();
	}

	/**
	 * 获取桶信息
	 * @param bucketName bucketName
	 * @return Bucket
	 */
	@GetMapping("/bucket/{bucketName}")
	public Bucket getBucket(@PathVariable @NotBlank String bucketName) {
		return ossTemplate.getBucket(bucketName)
			.orElseThrow(() -> new IllegalArgumentException("Bucket Name not found!"));
	}

	/**
	 * 删除桶
	 * @param bucketName bucketName
	 */
	@DeleteMapping("/bucket/{bucketName}")
	@ResponseStatus(HttpStatus.ACCEPTED)
	public void deleteBucket(@PathVariable @NotBlank String bucketName) {
		ossTemplate.removeBucket(bucketName);
	}

	/**
	 * 上传对象
	 * @param object MultipartFile
	 * @param bucketName bucketName
	 * @return S3ObjectSummary
	 * @throws IOException IOException
	 */
	@PostMapping("/object/{bucketName}")
	public S3ObjectSummary createObject(@RequestBody @NotNull MultipartFile object,
			@PathVariable @NotBlank String bucketName) throws IOException {

		InputStream inputStream = object.getInputStream();
		String name = object.getOriginalFilename();

		ossTemplate.putObject(bucketName, name, inputStream, object.getSize(), object.getContentType());
		S3Object objectInfo = ossTemplate.getObjectInfo(bucketName, name);
		ObjectMetadata objectMetadata = objectInfo.getObjectMetadata();
		S3ObjectSummary objectSummary = new S3ObjectSummary();
		objectSummary.setKey(objectInfo.getKey());
		objectSummary.setBucketName(bucketName);
		objectSummary.setETag(objectMetadata.getETag());
		objectSummary.setLastModified(objectMetadata.getLastModified());
		objectSummary.setSize(objectMetadata.getContentLength());
		return objectSummary;
	}

	/**
	 * 上传对象
	 * @param object MultipartFile
	 * @param bucketName bucketName
	 * @param objectName objectName
	 * @return S3ObjectSummary
	 * @throws IOException IOException
	 */
	@PostMapping("/object/{bucketName}/{objectName}")
	public S3ObjectSummary createObject(@RequestBody @NotNull MultipartFile object,
			@PathVariable @NotBlank String bucketName, @PathVariable @NotBlank String objectName) throws IOException {

		InputStream inputStream = object.getInputStream();
		ossTemplate.putObject(bucketName, objectName, inputStream, object.getSize(), object.getContentType());
		S3Object objectInfo = ossTemplate.getObjectInfo(bucketName, objectName);
		ObjectMetadata objectMetadata = objectInfo.getObjectMetadata();
		S3ObjectSummary objectSummary = new S3ObjectSummary();
		objectSummary.setKey(objectInfo.getKey());
		objectSummary.setBucketName(bucketName);
		objectSummary.setETag(objectMetadata.getETag());
		objectSummary.setLastModified(objectMetadata.getLastModified());
		objectSummary.setSize(objectMetadata.getContentLength());
		return objectSummary;
	}

	/**
	 * 根据桶获取对象
	 * @param bucketName bucketName
	 * @param objectName objectName
	 * @return List S3ObjectSummary
	 */
	@GetMapping("/object/{bucketName}/{objectName}")
	public List<S3ObjectSummary> filterObject(@PathVariable @NotBlank String bucketName,
			@PathVariable @NotBlank String objectName) {
		return ossTemplate.getAllObjectsByPrefix(bucketName, objectName);
	}

	/**
	 * 获取文件外链，只用于下载
	 * @param bucketName bucketName
	 * @param objectName objectName
	 * @param expires Integer
	 * @return Map String, Object
	 */
	@GetMapping("/object/{bucketName}/{objectName}/{expires}")
	public Map<String, Object> getObjectUrl(@PathVariable @NotBlank String bucketName,
			@PathVariable @NotBlank String objectName, @PathVariable @NotNull Integer expires) {
		Map<String, Object> responseBody = new HashMap<>(8);
		// Put Object info
		responseBody.put("bucket", bucketName);
		responseBody.put("object", objectName);
		responseBody.put("url", ossTemplate.getObjectURL(bucketName, objectName, expires));
		responseBody.put("expires", expires);
		return responseBody;
	}

	/**
	 * 获取文件上传外链，只用于上传
	 * @param bucketName bucketName
	 * @param objectName objectName
	 * @param expires Integer
	 * @return Map String, Object
	 */
	@GetMapping("/object/put/{bucketName}/{objectName}/{expires}")
	public Map<String, Object> getPutObjectUrl(@PathVariable @NotBlank String bucketName,
			@PathVariable @NotBlank String objectName, @PathVariable @NotNull Integer expires) {
		Map<String, Object> responseBody = new HashMap<>(8);
		// Put Object info
		responseBody.put("bucket", bucketName);
		responseBody.put("object", objectName);
		responseBody.put("url", ossTemplate.getPutObjectURL(bucketName, objectName, expires));
		responseBody.put("expires", expires);
		return responseBody;
	}

	/**
	 * 删除对象
	 * @param bucketName bucketName
	 * @param objectName objectName
	 */
	@ResponseStatus(HttpStatus.ACCEPTED)
	@DeleteMapping("/object/{bucketName}/{objectName}/")
	public void deleteObject(@PathVariable @NotBlank String bucketName, @PathVariable @NotBlank String objectName) {
		ossTemplate.removeObject(bucketName, objectName);
	}

}
