package cn.twelvet.multilevel.cache.support;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.connection.Message;
import org.springframework.data.redis.connection.MessageListener;
import org.springframework.data.redis.serializer.RedisSerializer;

import java.util.Objects;

/**
 * @author twelvet
 */
public class CacheMessageListener implements MessageListener {

	private final static Logger log = LoggerFactory.getLogger(RedisCaffeineCacheManager.class);

	/**
	 * RedisSerializer
	 */
	private final RedisSerializer<Object> redisSerializer;

	/**
	 * RedisCaffeineCacheManager
	 */
	private final RedisCaffeineCacheManager redisCaffeineCacheManager;

	/**
	 * CacheMessageListener
	 * @param redisSerializer RedisSerializer
	 * @param redisCaffeineCacheManager RedisCaffeineCacheManager
	 */
	public CacheMessageListener(RedisSerializer<Object> redisSerializer,
			RedisCaffeineCacheManager redisCaffeineCacheManager) {
		this.redisSerializer = redisSerializer;
		this.redisCaffeineCacheManager = redisCaffeineCacheManager;
	}

	/**
	 * getRedisSerializer
	 * @return RedisSerializer
	 */
	public RedisSerializer<Object> getRedisSerializer() {
		return redisSerializer;
	}

	/**
	 * getRedisCaffeineCacheManager
	 * @return RedisCaffeineCacheManager
	 */
	public RedisCaffeineCacheManager getRedisCaffeineCacheManager() {
		return redisCaffeineCacheManager;
	}

	/**
	 * onMessage
	 * @param message Message
	 * @param pattern byte
	 */
	@Override
	public void onMessage(Message message, byte[] pattern) {
		CacheMessage cacheMessage = (CacheMessage) redisSerializer.deserialize(message.getBody());
		if (!Objects.equals(cacheMessage.getServerId(), redisCaffeineCacheManager.getServerId())) {
			log.debug(
					"receive a redis topic message, clear local cache, the cacheName is {}, operation is {}, the key is {}",
					cacheMessage.getCacheName(), cacheMessage.getOperation(), cacheMessage.getKey());
			redisCaffeineCacheManager.clearLocal(cacheMessage.getCacheName(), cacheMessage.getKey(),
					cacheMessage.getOperation());
		}
	}

}
