package cn.twelvet.multilevel.cache.support;

import cn.twelvet.multilevel.cache.enums.CacheOperation;
import cn.twelvet.multilevel.cache.properties.CacheConfigProperties;
import com.github.benmanes.caffeine.cache.Caffeine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.Cache;
import org.springframework.data.redis.core.RedisTemplate;

import java.time.Duration;
import java.util.Collection;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Consumer;

/**
 * @author twelvet
 */
public class RedisCaffeineCacheManager implements CacheManager {

	private final static Logger log = LoggerFactory.getLogger(RedisCaffeineCacheManager.class);

	/**
	 * ConcurrentMap<String, Cache>
	 */
	private ConcurrentMap<String, Cache> cacheMap = new ConcurrentHashMap<>();

	/**
	 * CacheConfigProperties
	 */
	private CacheConfigProperties cacheConfigProperties;

	/**
	 * RedisTemplate<Object, Object>
	 */
	private RedisTemplate<Object, Object> stringKeyRedisTemplate;

	/**
	 * dynamic
	 */
	private boolean dynamic;

	/**
	 * cacheNames
	 */
	private Set<String> cacheNames;

	/**
	 * serverId
	 */
	private Object serverId;

	/**
	 * getCacheMap
	 * @return ConcurrentMap String, Cache
	 */
	public ConcurrentMap<String, Cache> getCacheMap() {
		return cacheMap;
	}

	/**
	 * setCacheMap
	 * @param cacheMap ConcurrentMap String, Cache
	 */
	public void setCacheMap(ConcurrentMap<String, Cache> cacheMap) {
		this.cacheMap = cacheMap;
	}

	/**
	 * getCacheConfigProperties
	 * @return CacheConfigProperties
	 */
	public CacheConfigProperties getCacheConfigProperties() {
		return cacheConfigProperties;
	}

	/**
	 * setCacheConfigProperties
	 * @param cacheConfigProperties CacheConfigProperties
	 */
	public void setCacheConfigProperties(CacheConfigProperties cacheConfigProperties) {
		this.cacheConfigProperties = cacheConfigProperties;
	}

	/**
	 * getStringKeyRedisTemplate
	 * @return RedisTemplate Object, Object
	 */
	public RedisTemplate<Object, Object> getStringKeyRedisTemplate() {
		return stringKeyRedisTemplate;
	}

	/**
	 * setStringKeyRedisTemplate
	 * @param stringKeyRedisTemplate RedisTemplate Object, Object
	 */
	public void setStringKeyRedisTemplate(RedisTemplate<Object, Object> stringKeyRedisTemplate) {
		this.stringKeyRedisTemplate = stringKeyRedisTemplate;
	}

	/**
	 * isDynamic
	 * @return boolean
	 */
	public boolean isDynamic() {
		return dynamic;
	}

	/**
	 * setDynamic
	 * @param dynamic boolean
	 */
	public void setDynamic(boolean dynamic) {
		this.dynamic = dynamic;
	}

	/**
	 * setDynamic
	 * @param cacheNames Set String
	 */
	public void setCacheNames(Set<String> cacheNames) {
		this.cacheNames = cacheNames;
	}

	/**
	 * getServerId
	 * @return Object
	 */
	public Object getServerId() {
		return serverId;
	}

	/**
	 * setServerId
	 * @param serverId Object
	 */
	public void setServerId(Object serverId) {
		this.serverId = serverId;
	}

	/**
	 * RedisCaffeineCacheManager
	 * @param cacheConfigProperties CacheConfigProperties
	 * @param stringKeyRedisTemplate RedisTemplate Object, Object
	 */
	public RedisCaffeineCacheManager(CacheConfigProperties cacheConfigProperties,
			RedisTemplate<Object, Object> stringKeyRedisTemplate) {
		super();
		this.cacheConfigProperties = cacheConfigProperties;
		this.stringKeyRedisTemplate = stringKeyRedisTemplate;
		this.dynamic = cacheConfigProperties.isDynamic();
		this.cacheNames = cacheConfigProperties.getCacheNames();
		this.serverId = cacheConfigProperties.getServerId();
	}

	/**
	 * getCache
	 * @param name String
	 * @return Cache
	 */
	@Override
	public Cache getCache(String name) {
		Cache cache = cacheMap.get(name);
		if (cache != null) {
			return cache;
		}
		if (!dynamic && !cacheNames.contains(name)) {
			return null;
		}

		cache = createCache(name);
		Cache oldCache = cacheMap.putIfAbsent(name, cache);
		log.debug("create cache instance, the cache name is : {}", name);
		return oldCache == null ? cache : oldCache;
	}

	/**
	 * getCaffeineCache
	 * @param name 缓存名
	 * @return Cache
	 * @param <K> k
	 * @param <V> v
	 */
	@Override
	@SuppressWarnings("unchecked")
	public <K, V> com.github.benmanes.caffeine.cache.Cache<K, V> getCaffeineCache(String name) {
		return (com.github.benmanes.caffeine.cache.Cache<K, V>) getCache(name);
	}

	/**
	 * createCache
	 * @param name String
	 * @return RedisCaffeineCache
	 */
	public RedisCaffeineCache createCache(String name) {
		return new RedisCaffeineCache(name, stringKeyRedisTemplate, caffeineCache(), cacheConfigProperties);
	}

	/**
	 * caffeineCache
	 * @return Cache
	 */
	public com.github.benmanes.caffeine.cache.Cache<Object, Object> caffeineCache() {
		return caffeineCacheBuilder().build();
	}

	/**
	 * caffeineCacheBuilder
	 * @return Caffeine Object, Object
	 */
	public Caffeine<Object, Object> caffeineCacheBuilder() {
		Caffeine<Object, Object> cacheBuilder = Caffeine.newBuilder();
		doIfPresent(cacheConfigProperties.getCaffeine().getExpireAfterAccess(), cacheBuilder::expireAfterAccess);
		doIfPresent(cacheConfigProperties.getCaffeine().getExpireAfterWrite(), cacheBuilder::expireAfterWrite);
		doIfPresent(cacheConfigProperties.getCaffeine().getRefreshAfterWrite(), cacheBuilder::refreshAfterWrite);
		if (cacheConfigProperties.getCaffeine().getInitialCapacity() > 0) {
			cacheBuilder.initialCapacity(cacheConfigProperties.getCaffeine().getInitialCapacity());
		}
		if (cacheConfigProperties.getCaffeine().getMaximumSize() > 0) {
			cacheBuilder.maximumSize(cacheConfigProperties.getCaffeine().getMaximumSize());
		}
		if (cacheConfigProperties.getCaffeine().getKeyStrength() != null) {
			switch (cacheConfigProperties.getCaffeine().getKeyStrength()) {
				case WEAK:
					cacheBuilder.weakKeys();
					break;
				case SOFT:
					throw new UnsupportedOperationException("caffeine 不支持 key 软引用");
				default:
			}
		}
		if (cacheConfigProperties.getCaffeine().getValueStrength() != null) {
			switch (cacheConfigProperties.getCaffeine().getValueStrength()) {
				case WEAK:
					cacheBuilder.weakValues();
					break;
				case SOFT:
					cacheBuilder.softValues();
					break;
				default:
			}
		}
		return cacheBuilder;
	}

	/**
	 * doIfPresent
	 * @param duration Duration
	 * @param consumer Consumer Duration
	 */
	protected static void doIfPresent(Duration duration, Consumer<Duration> consumer) {
		if (duration != null && !duration.isNegative()) {
			consumer.accept(duration);
		}
	}

	/**
	 * getCacheNames
	 * @return Collection String
	 */
	@Override
	public Collection<String> getCacheNames() {
		return this.cacheNames;
	}

	/**
	 * clearLocal
	 * @param cacheName String
	 * @param key Object
	 */
	public void clearLocal(String cacheName, Object key) {
		clearLocal(cacheName, key, CacheOperation.EVICT);
	}

	/**
	 * clearLocal
	 * @param cacheName String
	 * @param key Object
	 * @param operation CacheOperation
	 */
	@SuppressWarnings("unchecked")
	public void clearLocal(String cacheName, Object key, CacheOperation operation) {
		Cache cache = cacheMap.get(cacheName);
		if (cache == null) {
			return;
		}

		RedisCaffeineCache redisCaffeineCache = (RedisCaffeineCache) cache;
		if (CacheOperation.EVICT_BATCH.equals(operation)) {
			redisCaffeineCache.clearLocalBatch((Iterable<Object>) key);
		}
		else {
			redisCaffeineCache.clearLocal(key);
		}
	}

}
