package cn.twelvet.excel.aop;

import cn.twelvet.excel.annotation.RequestExcel;
import cn.twelvet.excel.converters.LocalDateStringConverter;
import cn.twelvet.excel.converters.LocalDateTimeStringConverter;
import cn.twelvet.excel.handler.request.ListAnalysisEventListener;
import cn.twelvet.excel.kit.ExcelException;
import com.alibaba.excel.EasyExcel;
import javax.servlet.http.HttpServletRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.core.MethodParameter;
import org.springframework.core.ResolvableType;
import org.springframework.lang.NonNull;
import org.springframework.ui.ModelMap;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.support.WebDataBinderFactory;
import org.springframework.web.context.request.NativeWebRequest;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.method.support.ModelAndViewContainer;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartRequest;

import java.io.InputStream;
import java.util.List;

/**
 * 上传excel 解析注解
 *
 * @author twelvet
 */
public class RequestExcelArgumentResolver implements HandlerMethodArgumentResolver {

	private final static Logger log = LoggerFactory.getLogger(RequestExcelArgumentResolver.class);

	/**
	 * 判断是否存在参数被RequestExcel注解
	 * @param parameter MethodParameter
	 * @return boolean
	 */
	@Override
	public boolean supportsParameter(MethodParameter parameter) {
		return parameter.hasParameterAnnotation(RequestExcel.class);
	}

	/**
	 * 处理RequestExcel注解解析参数
	 * @param methodParameter MethodParameter
	 * @param modelAndViewContainer ModelAndViewContainer
	 * @param nativeWebRequest NativeWebRequest
	 * @param webDataBinderFactory WebDataBinderFactory
	 * @return Object
	 */
	@Override
	public Object resolveArgument(@NonNull MethodParameter methodParameter, ModelAndViewContainer modelAndViewContainer,
			@NonNull NativeWebRequest nativeWebRequest, WebDataBinderFactory webDataBinderFactory) {
		try {
			Class<?> parameterType = methodParameter.getParameterType();
			// 检查是否为List
			if (!parameterType.isAssignableFrom(List.class)) {
				throw new IllegalArgumentException(
						"Excel upload request resolver error, @RequestExcel parameter is not List " + parameterType);
			}

			// 处理自定义 readListener
			RequestExcel requestExcel = methodParameter.getParameterAnnotation(RequestExcel.class);
			assert requestExcel != null;
			Class<? extends ListAnalysisEventListener<?>> readListenerClass = requestExcel.readListener();
			ListAnalysisEventListener<?> readListener = BeanUtils.instantiateClass(readListenerClass);
			// 获取请求文件流
			HttpServletRequest request = nativeWebRequest.getNativeRequest(HttpServletRequest.class);
			assert request != null;
			InputStream inputStream;
			if (request instanceof MultipartRequest) {
				MultipartFile file = ((MultipartRequest) request).getFile(requestExcel.fileName());
				assert file != null;
				inputStream = file.getInputStream();
			}
			else {
				inputStream = request.getInputStream();
			}

			// 获取注入Class类型
			Class<?> excelModelClass = ResolvableType.forMethodParameter(methodParameter).getGeneric(0).resolve();

			// 这里需要指定读用哪个 class 去读，然后读取第一个 sheet 文件流会自动关闭
			EasyExcel.read(inputStream, excelModelClass, readListener)
				.registerConverter(LocalDateStringConverter.INSTANCE)
				.registerConverter(LocalDateTimeStringConverter.INSTANCE)
				.ignoreEmptyRow(requestExcel.ignoreEmptyRow())
				.sheet()
				.headRowNumber(requestExcel.headRowNumber())
				.doRead();

			// 校验失败的数据处理 交给 BindResult
			WebDataBinder dataBinder = webDataBinderFactory.createBinder(nativeWebRequest, readListener.getErrors(),
					"excel");
			ModelMap model = modelAndViewContainer.getModel();
			model.put(BindingResult.MODEL_KEY_PREFIX + "excel", dataBinder.getBindingResult());

			return readListener.getList();
		}
		catch (Exception e) {
			log.error("导入Excel失败", e);
			throw new ExcelException("导入Excel失败");
		}
	}

}
