package cn.twelvet.excel.aop;

import cn.twelvet.excel.annotation.ResponseExcel;
import cn.twelvet.excel.processor.NameProcessor;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import java.time.LocalDateTime;
import java.util.Objects;

/**
 * Excel名称解析处理切面
 *
 * @author twelvet
 */
@Aspect
public class DynamicNameAspect {

	/**
	 * Excel Key
	 */
	public static final String EXCEL_NAME_KEY = "__EXCEL_NAME_KEY__";

	/**
	 * SPL解析器
	 */
	private final NameProcessor processor;

	/**
	 * DynamicNameAspect
	 * @param processor NameProcessor
	 */
	public DynamicNameAspect(NameProcessor processor) {
		this.processor = processor;
	}

	/**
	 * around
	 * @param point JoinPoint
	 * @param responseExcel ResponseExcel
	 */
	@Before("@annotation(responseExcel)")
	public void around(JoinPoint point, ResponseExcel responseExcel) {
		MethodSignature ms = (MethodSignature) point.getSignature();

		String name = responseExcel.name();
		// 当配置的 excel 名称为空时，取当前时间
		if (!StringUtils.hasText(name)) {
			name = LocalDateTime.now().toString();
		}
		else {
			name = processor.doDetermineName(point.getArgs(), ms.getMethod(), responseExcel.name());
		}

		RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
		Objects.requireNonNull(requestAttributes).setAttribute(EXCEL_NAME_KEY, name, RequestAttributes.SCOPE_REQUEST);
	}

}
