package cn.twelvet.excel;

import cn.twelvet.excel.aop.DynamicNameAspect;
import cn.twelvet.excel.aop.RequestExcelArgumentResolver;
import cn.twelvet.excel.aop.ResponseExcelReturnValueHandler;
import cn.twelvet.excel.config.ExcelConfigProperties;
import cn.twelvet.excel.processor.NameProcessor;
import cn.twelvet.excel.processor.impl.NameSpelExpressionProcessorImpl;
import javax.annotation.PostConstruct;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.method.support.HandlerMethodReturnValueHandler;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerAdapter;

import java.util.ArrayList;
import java.util.List;

/**
 * 自动配置入口
 *
 * @author twelvet
 */
@AutoConfiguration
@Import(ExcelHandlerConfiguration.class)
@EnableConfigurationProperties(ExcelConfigProperties.class)
public class ResponseExcelAutoConfiguration {

	private final RequestMappingHandlerAdapter requestMappingHandlerAdapter;

	private final ResponseExcelReturnValueHandler responseExcelReturnValueHandler;

	/**
	 * IOC
	 * @param requestMappingHandlerAdapter RequestMappingHandlerAdapter
	 * @param responseExcelReturnValueHandler ResponseExcelReturnValueHandler
	 */
	public ResponseExcelAutoConfiguration(RequestMappingHandlerAdapter requestMappingHandlerAdapter,
			ResponseExcelReturnValueHandler responseExcelReturnValueHandler) {
		this.requestMappingHandlerAdapter = requestMappingHandlerAdapter;
		this.responseExcelReturnValueHandler = responseExcelReturnValueHandler;
	}

	/**
	 * SPEL 解析处理器
	 * @return NameProcessor excel名称解析器
	 */
	@Bean
	@ConditionalOnMissingBean
	public NameProcessor nameProcessor() {
		return new NameSpelExpressionProcessorImpl();
	}

	/**
	 * Excel名称解析处理切面
	 * @param nameProcessor SPEL 解析处理器
	 * @return DynamicNameAspect
	 */
	@Bean
	@ConditionalOnMissingBean
	public DynamicNameAspect dynamicNameAspect(NameProcessor nameProcessor) {
		return new DynamicNameAspect(nameProcessor);
	}

	/**
	 * 追加 Excel返回值处理器 到 springmvc 中
	 */
	public void addReturnValueHandlers() {
		List<HandlerMethodReturnValueHandler> returnValueHandlers = requestMappingHandlerAdapter
			.getReturnValueHandlers();
		List<HandlerMethodReturnValueHandler> handlerMethodReturnValueHandlers = new ArrayList<>();
		handlerMethodReturnValueHandlers.add(responseExcelReturnValueHandler);
		assert returnValueHandlers != null;
		handlerMethodReturnValueHandlers.addAll(returnValueHandlers);
		requestMappingHandlerAdapter.setReturnValueHandlers(handlerMethodReturnValueHandlers);
	}

	/**
	 * 追加 Excel 请求处理器 到 springmvc 中
	 */
	public void addRequestExcelArgumentResolver() {
		List<HandlerMethodArgumentResolver> argumentResolvers = requestMappingHandlerAdapter.getArgumentResolvers();
		List<HandlerMethodArgumentResolver> handlerMethodArgumentResolvers = new ArrayList<>();
		handlerMethodArgumentResolvers.add(new RequestExcelArgumentResolver());
		assert argumentResolvers != null;
		handlerMethodArgumentResolvers.addAll(argumentResolvers);
		requestMappingHandlerAdapter.setArgumentResolvers(handlerMethodArgumentResolvers);
	}

	/**
	 * add request and response handlers
	 */
	@PostConstruct
	public void addHandlers() {
		// add response
		addReturnValueHandlers();
		// add request
		addRequestExcelArgumentResolver();
	}

}
