package cn.twelvet.excel;

import cn.twelvet.excel.aop.ResponseExcelReturnValueHandler;
import cn.twelvet.excel.config.ExcelConfigProperties;
import cn.twelvet.excel.enhance.impl.DefaultWriterBuilderEnhancerImpl;
import cn.twelvet.excel.enhance.WriterBuilderEnhancer;
import cn.twelvet.excel.handler.response.impl.ManySheetWriteHandlerImpl;
import cn.twelvet.excel.handler.response.SheetWriteHandler;
import cn.twelvet.excel.handler.response.impl.SingleSheetWriteHandlerImpl;
import cn.twelvet.excel.head.impl.I18nHeaderCellWriteHandler;
import com.alibaba.excel.converters.Converter;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Bean;

import java.util.List;

/**
 * 装载Bean
 *
 * @author twelvet
 */
public class ExcelHandlerConfiguration {

	private final ExcelConfigProperties configProperties;

	private final ObjectProvider<List<Converter<?>>> converterProvider;

	/**
	 * IOC
	 * @param configProperties ExcelConfigProperties
	 * @param converterProvider ObjectProvider
	 */
	public ExcelHandlerConfiguration(ExcelConfigProperties configProperties,
			ObjectProvider<List<Converter<?>>> converterProvider) {
		this.configProperties = configProperties;
		this.converterProvider = converterProvider;
	}

	/**
	 * ExcelBuild增强
	 * @return DefaultWriterBuilderEnhancer 默认什么也不做的增强器
	 */
	@Bean
	@ConditionalOnMissingBean
	public WriterBuilderEnhancer writerBuilderEnhancer() {
		return new DefaultWriterBuilderEnhancerImpl();
	}

	/**
	 * 单sheet 写入处理器
	 * @return SingleSheetWriteHandler
	 */
	@Bean
	@ConditionalOnMissingBean
	public SingleSheetWriteHandlerImpl singleSheetWriteHandler() {
		return new SingleSheetWriteHandlerImpl(configProperties, converterProvider, writerBuilderEnhancer());
	}

	/**
	 * 多sheet 写入处理器
	 * @return ManySheetWriteHandler
	 */
	@Bean
	@ConditionalOnMissingBean
	public ManySheetWriteHandlerImpl manySheetWriteHandler() {
		return new ManySheetWriteHandlerImpl(configProperties, converterProvider, writerBuilderEnhancer());
	}

	/**
	 * 返回Excel文件的 response 处理器
	 * @param sheetWriteHandlerList 页签写入处理器集合
	 * @return ResponseExcelReturnValueHandler
	 */
	@Bean
	@ConditionalOnMissingBean
	public ResponseExcelReturnValueHandler responseExcelReturnValueHandler(
			List<SheetWriteHandler> sheetWriteHandlerList) {
		return new ResponseExcelReturnValueHandler(sheetWriteHandlerList);
	}

	/**
	 * excel 头的国际化处理器
	 * @param messageSource 国际化源
	 * @return I18nHeaderCellWriteHandler
	 */
	@Bean
	@ConditionalOnBean(MessageSource.class)
	@ConditionalOnMissingBean
	public I18nHeaderCellWriteHandler i18nHeaderCellWriteHandler(MessageSource messageSource) {
		return new I18nHeaderCellWriteHandler(messageSource);
	}

}
