package cn.meteor.common.core.yaml;// Copyright (C), Created on 2021-04-02

import cn.hutool.core.util.StrUtil;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.dataformat.javaprop.JavaPropsFactory;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.fasterxml.jackson.dataformat.yaml.YAMLGenerator;
import com.fasterxml.jackson.dataformat.yaml.YAMLParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.util.Properties;

/**
 * Yaml 和 Properties 互相转化
 *
 * @author ths
 * @since 1.0.0
 */
public class TransferUtil {

	private static final Logger log = LoggerFactory.getLogger(TransferUtil.class);

	/**
	 * Yaml 转 Properties
	 *
	 * @param path 文件路径
	 * @return Properties
	 */
	public static Properties yaml2Properties(String path) {
		return yaml2Properties(path, true);
	}

	/**
	 * Yaml 转 Properties
	 *
	 * @param path      文件路径
	 * @param classpath 是否是classpath路径，否则使用物理方式读取文件
	 * @return Properties
	 */
	public static Properties yaml2Properties(String path, boolean classpath) {
		Properties props = new Properties();
		YAMLParser parser = null;
		try {
			YAMLFactory yamlFactory = new YAMLFactory();
			if (classpath) {
				parser = yamlFactory.createParser(TransferUtil.class.getClassLoader().getResourceAsStream(path));
			} else {
				parser = yamlFactory.createParser(new InputStreamReader(new FileInputStream(path), StandardCharsets.UTF_8));
			}
			StringBuilder key = new StringBuilder();
			String value;
			JsonToken token = parser.nextToken();
			while (token != null) {
				if (JsonToken.FIELD_NAME.equals(token)) {
					if (key.length() > 0) {
						key.append(StrUtil.DOT);
					}
					key.append(parser.getCurrentName());

					token = parser.nextToken();
					if (JsonToken.START_OBJECT.equals(token)) {
						continue;
					}
					value = parser.getText();
					props.setProperty(key.toString(), value);

					int dotOffset = key.toString().lastIndexOf(StrUtil.DOT);
					if (dotOffset > 0) {
						key = new StringBuilder(key.substring(0, dotOffset));
					}
				} else if (JsonToken.END_OBJECT.equals(token)) {
					int dotOffset = key.toString().lastIndexOf(StrUtil.DOT);
					if (dotOffset > 0) {
						key = new StringBuilder(key.substring(0, dotOffset));
					} else {
						key = new StringBuilder();
					}
				}
				token = parser.nextToken();
			}
		} catch (Exception e) {
			log.error("Yaml 转 Properties时异常", e);
		} finally {
			if (parser != null) {
				try {
					parser.close();
				} catch (IOException ignored) {
				}
			}
		}
		return props;
	}

	/**
	 * Properties时异常 转 Yaml
	 *
	 * @param path       原物理文件
	 * @param targetPath 目标物理文件
	 * @throws IOException IO异常
	 */
	public static void properties2Yaml(String path, String targetPath) throws IOException {
		JsonParser parser;
		JavaPropsFactory factory = new JavaPropsFactory();
		parser = factory.createParser(
			new InputStreamReader(new FileInputStream(path), StandardCharsets.UTF_8));

		YAMLFactory yamlFactory = new YAMLFactory();
		YAMLGenerator generator = yamlFactory.createGenerator(
			new OutputStreamWriter(new FileOutputStream(targetPath), StandardCharsets.UTF_8));
		JsonToken token = parser.nextToken();

		while (token != null) {
			if (JsonToken.START_OBJECT.equals(token)) {
				generator.writeStartObject();
			} else if (JsonToken.FIELD_NAME.equals(token)) {
				generator.writeFieldName(parser.getCurrentName());
			} else if (JsonToken.VALUE_STRING.equals(token)) {
				generator.writeString(parser.getText());
			} else if (JsonToken.END_OBJECT.equals(token)) {
				generator.writeEndObject();
			}
			token = parser.nextToken();
		}
	}

}
