package cn.meteor.common.web.advice;// Copyright (C), Created on 2021-04-04

import cn.meteor.common.enums.ResultType;
import cn.meteor.common.exception.ServiceException;
import cn.meteor.common.model.R;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.converter.HttpMessageConversionException;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;

import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.io.IOException;
import java.net.SocketTimeoutException;
import java.util.Optional;

/**
 * 全局异常处理
 *
 * @author ths
 * @since 1.0.0
 */
@Slf4j
@RestControllerAdvice
public class GlobalExceptionHandler {

	/**
	 * 参数校验错误拦截处理
	 *
	 * @param e 错误信息集合
	 * @return 错误信息
	 */
	@ExceptionHandler(value = {MethodArgumentNotValidException.class, BindException.class, ConstraintViolationException.class})
	public R<Object> failure(Exception e) {
		BindingResult result = null;
		if (e instanceof MethodArgumentNotValidException) {
			result = ((MethodArgumentNotValidException) e).getBindingResult();
		} else if (e instanceof BindException) {
			result = ((BindException) e).getBindingResult();
			// MethodValidationInterceptor
		} else if (e instanceof ConstraintViolationException) {
			Optional<ConstraintViolation<?>> first = ((ConstraintViolationException) e).getConstraintViolations().stream().findFirst();
			return R.failure(first.isPresent() ? first.get().getMessage() : e.getMessage());
		}
		if (result == null) {
			return R.failure(log, ResultType.SERVER_ERROR, e);
		}
		// 返回错误信息
		return result.hasErrors() ? R.failure(result.getAllErrors().get(0).getDefaultMessage()) : R.success();
	}

	/**
	 * 通用异常处理
	 *
	 * @param e 错误
	 * @param response 响应体
	 * @return 错误信息
	 */
	@ExceptionHandler({Exception.class, Throwable.class, Error.class, IOException.class, RuntimeException.class})
	public R<Object> failure(Exception e, HttpServletResponse response) {
		// 设置 HTTP 状态码
		response.setStatus(HttpStatus.OK.value());
		R<Object> result;
		// 判断是否为 ServiceException 异常
		if (e instanceof ServiceException) {
			// 返回错误信息
			result = R.failure((ServiceException) e);
		}
		// 判断是否为 请求方式限制
		else if (e instanceof HttpRequestMethodNotSupportedException) {
			result = R.failure(((HttpRequestMethodNotSupportedException) e).getMethod() + "请求方式不被支持");
		}
		// 判断是否为 丢失请求参数异常
		else if (e instanceof MissingServletRequestParameterException) {
			result = R.failure(e.getMessage());
		}
		// 判断是否为 缺少请求体异常
		else if (e instanceof HttpMessageNotReadableException) {
			result = R.failure(ResultType.REQUEST_MISSING_BODY_ERROR);
		}
		// 判断是否为 Content-Type参数设置错误
		else if (e instanceof HttpMediaTypeNotSupportedException) {
			result = R.failure(ResultType.CONTENT_TYPE_ERROR);
		}
		// 判断是否为 Accept参数错误错误
		else if (e instanceof MethodArgumentTypeMismatchException) {
			result = R.failure(ResultType.ACCEPT_ERROR);
		}
		// 判断是否为 Accept参数设置错误
		else if (e instanceof HttpMediaTypeNotAcceptableException) {
			result = R.failure(ResultType.NOT_ACCEPT_ERROR);
		}
		// 判断是否为 请求参数错误异常
		else if (e instanceof MethodArgumentNotValidException) {
			result = R.failure(ResultType.PARAMETER_ERROR);
		}
		// 判断是否为 参数类型转换错误
		else if (e instanceof HttpMessageConversionException) {
			result = R.failure(log, ResultType.MESSAGE_CONVERT, e);
		}
		// 判断是否为 服务器通讯超时异常
		else if (e instanceof SocketTimeoutException) {
			result = R.failure(log, ResultType.SOCKET_TIMEOUT, e);
		} else if (e instanceof IllegalArgumentException) {
			result = R.failure(e.getMessage());
		}
		// 不知道异常原因，默认返回未知异常
		else {
			result = R.failure(log, ResultType.SERVER_ERROR, e);
		}
		return result;
	}

}
