package cn.meteor.common.web.util;

import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.HashMap;
import java.util.Map;

/**
 * Http 工具类
 *
 * @author ths
 * @since 1.0.0
 */
public class RestTemplateUtil {

	public final static RestTemplate REST = new RestTemplate();

	// ----------------------------------GET-------------------------------------------------------

	/**
	 * get请求
	 *
	 * @param url 请求地址
	 * @return 响应
	 */
	public static String get(String url) {
		return get(url, null, String.class);
	}

	/**
	 * get请求
	 *
	 * @param url    请求地址
	 * @param reqMap 请求信息
	 * @return 响应
	 */
	public static String get(String url, Map<String, Object> reqMap) {
		return get(url, null, reqMap, String.class);
	}

	/**
	 * get请求
	 *
	 * @param url  请求地址
	 * @param resp 响应结果
	 * @param <T>  响应结果类型
	 * @return 响应
	 */
	public static <T> T get(String url, Class<T> resp) {
		return get(url, null, resp);
	}

	/**
	 * get请求
	 *
	 * @param url       请求地址
	 * @param headerMap 头部信息
	 * @param resp      响应结果
	 * @param <T>       响应结果类型
	 * @return 响应
	 */
	public static <T> T get(String url, Map<String, String> headerMap, Class<T> resp) {
		return get(url, headerMap, null, resp);
	}

	/**
	 * get请求
	 *
	 * @param url       请求地址
	 * @param headerMap 头部信息
	 * @param reqMap    请求信息
	 * @param resp      响应结果
	 * @param <T>       响应结果类型
	 * @return 响应
	 */
	public static <T> T get(String url, Map<String, String> headerMap, Map<String, Object> reqMap, Class<T> resp) {
		String reallyUrl = url;
		if (reqMap != null) {
			UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(url);
			for (Map.Entry<String, Object> entry : reqMap.entrySet()) {
				builder.queryParam(entry.getKey(), entry.getValue());
			}
			reallyUrl = builder.build().toString();
		}
		return exchange(reallyUrl, HttpMethod.POST, headerMap, null, resp);
	}

	// ----------------------------------POST-------------------------------------------------------

	/**
	 * post 请求
	 *
	 * @param url    请求地址
	 * @param reqMap 请求参数
	 * @return 响应
	 */
	public static String post(String url, Map<String, Object> reqMap) {
		return post(url, reqMap, String.class);
	}

	/**
	 * post 请求
	 *
	 * @param url       请求地址
	 * @param headerMap 头信息
	 * @param reqMap    请求信息
	 * @return 响应
	 */
	public static String post(String url, Map<String, String> headerMap, Map<String, Object> reqMap) {
		return post(url, headerMap, reqMap, String.class);
	}

	/**
	 * post 请求
	 *
	 * @param url    请求地址
	 * @param reqMap 请求信息
	 * @param resp   响应结果类型
	 * @param <T>    响应结果类型
	 * @return 响应
	 */
	public static <T> T post(String url, Map<String, Object> reqMap, Class<T> resp) {
		return post(url, null, reqMap, resp);
	}

	/**
	 * post 请求
	 *
	 * @param url       请求地址
	 * @param headerMap 头信息
	 * @param reqMap    请求信息
	 * @param resp      响应结果类型
	 * @param <T>       响应结果类型
	 * @return 响应
	 */
	public static <T> T post(String url, Map<String, String> headerMap, Map<String, Object> reqMap, Class<T> resp) {
		return exchange(url, HttpMethod.POST, headerMap, reqMap, resp);
	}


	/**
	 * 通用调用方式
	 *
	 * @param url       请求URL
	 * @param method    请求方法类型
	 * @param headerMap 头信息
	 * @param reqMap    请求信息
	 * @param resp      响应结果类型
	 * @param <T>       响应结果类型
	 * @return ResponseEntity 响应对象封装类
	 */
	public static <T> T exchange(String url, HttpMethod method, Map<String, String> headerMap, Map<String, Object> reqMap, Class<T> resp) throws RestClientException {
		HttpHeaders httpHeaders = new HttpHeaders();
		if (headerMap != null) {
			for (Map.Entry<String, String> stringStringEntry : headerMap.entrySet()) {
				httpHeaders.add(stringStringEntry.getKey(), stringStringEntry.getValue());
			}
		}
		if (reqMap == null) {
			reqMap = new HashMap<>(8);
		}
		HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(reqMap, httpHeaders);
		ResponseEntity<T> result = REST.exchange(url, method, httpEntity, resp);
		return result.getBody();
	}

}
