package cn.meteor.common.redis.dao;

import org.springframework.data.redis.connection.DataType;
import org.springframework.data.redis.core.ZSetOperations.TypedTuple;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @author ths
 */
@SuppressWarnings("all")
public interface IRedisDao<K, V> {

	/**
	 * 获取 db 数量
	 *
	 * @return db 数量
	 */
	int getSize();

	/**
	 * 返回key所储存的值的类型
	 *
	 * @param key 键 键
	 * @return DataType[NONE(" none "), STRING("string"), LIST("list"), SET("set"), ZSET("zset"), HASH("hash");]
	 */
	DataType type(K key);

	/**
	 * 判断缓存中是否有对应的value
	 *
	 * @param key 键
	 * @return boolean
	 */
	boolean existsKey(K key);

	/**
	 * 重命名 key name
	 *
	 * @param oldKey 原有key名
	 * @param newKey 新key名
	 */
	void renameKey(K oldKey, K newKey);

	/**
	 * 根据key获取key列表(key值可为模糊匹配 a:b:*  | *代表任意字符)
	 *
	 * @param pattern 表达式
	 * @return 键集合
	 */
	Set<K> keys(K pattern);

	/**
	 * 根据key获取个数(key值可为模糊匹配 a:b:*  | *代表任意字符)
	 *
	 * @param pattern 键
	 * @return int
	 */
	int count(K pattern);

	/**
	 * 根据key删除对应的value
	 *
	 * @param key 键
	 * @return boolean
	 */
	boolean delete(K key);

	/**
	 * 批量删除对应的value
	 *
	 * @param keys 键集合
	 */
	void delete(K[] keys);

	/**
	 * 批量删除key(key值可为模糊匹配 a:b:*  | *代表任意字符)
	 *
	 * @param pattern 表达式
	 * @return boolean
	 */
	long deletePattern(K pattern);

	/**
	 * 批量删除对应的value
	 *
	 * @param keys 键集合
	 * @return boolean
	 */
	long delete(Set<K> keys);

	/**
	 * 写入缓存(操作字符串)
	 *
	 * @param key   键
	 * @param value 值
	 * @return boolean
	 */
	boolean set(K key, V value);

	/**
	 * 将key中储存的数字加上指定的增量值(操作字符串)
	 *
	 * @param key   键
	 * @param value 值
	 * @return boolean
	 */
	boolean incr(K key, Long value);

	/**
	 * 写入缓存设置时效时间(操作字符串)
	 *
	 * @param key        键
	 * @param value      值
	 * @param expireTime 过期时间（秒）
	 * @param unit       时间单位
	 * @return boolean
	 */
	boolean set(K key, V value, Long expireTime, TimeUnit unit);

	/**
	 * 写入缓存设置时效时间(操作字符串)
	 *
	 * @param key        键
	 * @param value      值
	 * @param expireTime 过期时间（秒）
	 * @return boolean
	 */
	boolean set(K key, V value, Long expireTime);

	/**
	 * 更新写入缓存设置时效时间(操作字符串)
	 *
	 * @param key        键
	 * @param expireTime 过期时间（秒）
	 * @return boolean
	 */
	boolean updateExpireTime(K key, Long expireTime);

	/**
	 * 读取缓存(操作字符串)
	 *
	 * @param key   键
	 * @param clazz 转化类型
	 * @param <T>   返回类泛型
	 * @return 对象
	 */
	<T> T get(K key, Class<T> clazz);

	/**
	 * 哈希 添加(操作hash)
	 *
	 * @param key     键
	 * @param hashKey 哈希中的键
	 * @param value   哈希中的值
	 */
	void hSet(K key, K hashKey, V value);

	/**
	 * 将hashKey中储存的数字加上指定的增量值(操作hash)
	 *
	 * @param key     键
	 * @param hashKey 哈希中的键
	 * @param value   哈希中的值
	 */
	void hIncr(K key, K hashKey, Long value);

	/**
	 * 哈希 添加(操作hash)
	 *
	 * @param key 键
	 * @param map 集合
	 */
	void hSetAll(K key, Map<K, V> map);

	/**
	 * 哈希获取数据(操作hash)
	 *
	 * @param key 键
	 * @return Map
	 */
	Map<K, V> hGet(K key);

	/**
	 * 哈希获取数据(操作hash)
	 *
	 * @param key     键
	 * @param hashKey 哈希中的键
	 * @param clazz   类型转化
	 * @param <T>     返回类泛型
	 * @return 哈希中的值
	 */
	<T> T hGet(K key, K hashKey, Class<T> clazz);

	/**
	 * 哈希删除数据(操作hash)
	 *
	 * @param key     键
	 * @param hashKey 哈希中的键
	 * @return 删除数量
	 */
	long hDel(K key, K... hashKey);

	/**
	 * 获取key所对应的散列表的大小个数(操作hash)
	 *
	 * @param key 键
	 * @return 数量
	 */
	long hSize(K key);

	/**
	 * 获取key获取哈希表所有value
	 *
	 * @param key 键
	 * @return 集合
	 */
	List<V> hValues(K key);

	/**
	 * 获取列表中个数
	 *
	 * @param key 键
	 * @return long
	 */
	long lSize(K key);

	/**
	 * 通过其索引从列表获取第一个元素(操作list)
	 *
	 * @param key 键
	 * @return V
	 */
	V lIndexFirst(K key);

	/**
	 * 通过其索引从列表获取元素(操作list)
	 *
	 * @param key             键
	 * @param index:索引位置,从0开始
	 * @return V
	 */
	V lIndex(K key, long index);

	/**
	 * 从左向右添加列表(操作list)
	 *
	 * @param key   键
	 * @param value 值
	 * @return boolean
	 */
	boolean lLeftPush(K key, V value);

	/**
	 * 从左向右添加列表(操作list);如果bool=true,会删除列表中已经存在的数据,然后再进行添加(仅针对字符串列表,其它待测)
	 *
	 * @param key   键
	 * @param value 值
	 * @param bool  如果bool=true,会删除列表中已经存在的数据
	 * @return boolean
	 */
	boolean lLeftPush(K key, V value, boolean bool);

	/**
	 * 从左向右添加列表(操作list)
	 *
	 * @param key 键
	 * @param lst 集合
	 * @return 数量
	 */
	long lLeftPushAll(K key, List<V> lst);

	/**
	 * 从右向左添加列表(操作list)
	 *
	 * @param key   键
	 * @param value 值
	 * @return boolean
	 */
	boolean lRightPush(K key, V value);

	/**
	 * 从右向左添加列表(操作list);如果bool=true,会删除列表中已经存在的数据,然后再进行添加(仅针对字符串列表,其它待测)
	 *
	 * @param key   键
	 * @param value 值
	 * @param bool  如果bool=true,会删除列表中已经存在的数据
	 * @return boolean
	 */
	boolean lRightPush(K key, V value, boolean bool);

	/**
	 * 从右向左添加列表(操作list)
	 *
	 * @param key 键
	 * @param lst 集合
	 * @return 数量
	 */
	long lRightPushAll(K key, List<V> lst);

	/**
	 * 删除并获取列表中的第1个元素(操作list)
	 *
	 * @param key 键
	 * @return V
	 */
	V lLeftPop(K key);

	/**
	 * 删除并获取列表中的最后1个元素(操作list)
	 *
	 * @param key 键
	 * @return V
	 */
	V lRightPop(K key);

	/**
	 * 移除k中的count个,返回删除的个数；如果没有这个元素则返回0(操作list)
	 *
	 * @param key   键
	 * @param count 数量
	 * @return 删除的数量
	 */
	long lRemove(K key, long count);

	/**
	 * 移除k中值为v的count个,返回删除的个数；如果没有这个元素则返回0(操作list)
	 *
	 * @param key   键
	 * @param count 数量
	 * @param value 值
	 * @return 删除的数量
	 */
	long lRemove(K key, long count, V value);

	/**
	 * 移除k中值为v的所有数据,返回删除的个数；如果没有这个元素则返回0(操作list)
	 *
	 * @param key   键
	 * @param value 值
	 * @return 删除的数量
	 */
	long lRemove(K key, V value);

	/**
	 * 根据key获取获取List列表(操作list)
	 *
	 * @param key 键
	 * @return 集合
	 */
	List<V> lRange(K key);

	/**
	 * 根据key获取列表中第start至end的数据(操作list)
	 *
	 * @param key   键
	 * @param start 开始位置
	 * @param end   结束位置
	 * @return 集合
	 */
	List<V> lRange(K key, long start, long end);

	/**
	 * 集合添加
	 *
	 * @param key   键
	 * @param value 值
	 * @return 添加的数量
	 */
	long sAdd(K key, V... value);

	/**
	 * 返回集合中的一个随机元素(不删除)
	 *
	 * @param key 键
	 * @return V
	 */
	V sRandomMember(K key);

	/**
	 * 获取多个key无序集合中的元素，count表示个数
	 *
	 * @param key   键
	 * @param count 数量
	 * @return 集合
	 */
	List<V> sRandomMembers(K key, long count);

	/**
	 * 获取多个key无序集合中的元素(去重),count表示个数
	 *
	 * @param key   键
	 * @param count 数量
	 * @return V
	 */
	Set<V> sDistinctRandomMembers(K key, long count);

	/**
	 * 移除并返回集合中的一个随机元素
	 *
	 * @param key 键
	 * @return V
	 */
	V sPop(K key);

	/**
	 * 删除集合中的指定元素
	 *
	 * @param key    键
	 * @param values 元素集
	 * @return 删除个数
	 */
	long sRemove(K key, V... values);

	/**
	 * 获取集合
	 *
	 * @param key 键
	 * @return 集合
	 */
	Set<V> sMembers(K key);

	/**
	 * 集合是否存在value值
	 *
	 * @param key   键
	 * @param value 值
	 * @return boolean
	 */
	boolean sIsMember(K key, V value);

	/**
	 * 获取集合长度
	 *
	 * @param key 键
	 * @return long
	 */
	long sSize(K key);

	/**
	 * key无序集合与otherKey无序集合的差集
	 *
	 * @param key      键
	 * @param otherKey 另外的键
	 * @return 集合
	 */
	Set<V> sDifference(K key, K otherKey);

	/**
	 * 有序集合添加
	 *
	 * @param key   键
	 * @param value 值
	 * @param score 权重、优先级
	 * @return boolean
	 */
	boolean zAdd(K key, V value, double score);

	/**
	 * 有序集合添加
	 *
	 * @param key    键
	 * @param tuples 有序集合
	 * @return 添加数量
	 */
	long zAdd(K key, Set<TypedTuple<V>> tuples);

	/**
	 * 有序集合删除
	 *
	 * @param key    键
	 * @param values 值
	 * @return 删除数量
	 */
	long zRemove(K key, V... values);

	/**
	 * 根据键和权重范围获取有序集合
	 *
	 * @param key       键
	 * @param lowScore  低权重、优先级
	 * @param highScore 高权重、优先级
	 * @return 集合
	 */
	Set<V> zRangeByScore(K key, double lowScore, double highScore);
}
