package cn.meteor.common.redis.configure;

import cn.meteor.common.redis.lock.DistributedLocker;
import cn.meteor.common.redis.lock.RedissLockUtil;
import cn.meteor.common.redis.lock.RedissonDistributedLocker;
import org.redisson.Redisson;
import org.redisson.api.RedissonClient;
import org.redisson.config.Config;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.data.redis.RedisProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;

/**
 * RedissonConfigure
 *
 * @author ths
 */
@EnableConfigurationProperties(RedisProperties.class)
@ConditionalOnClass(RedissonClient.class)
public class RedissonConfigure {

	@Autowired
	private RedisProperties redisProperties;

	@Bean
	public RedissonClient redissonClient() {
		Config config = new Config();
		// 集群模式
		if (redisProperties.getCluster() != null) {
			String[] clusterNodes = new String[redisProperties.getCluster().getNodes().size()];
			for (int i = 0; i < redisProperties.getCluster().getNodes().size(); i++) {
				clusterNodes[i] = "redis://" + redisProperties.getCluster().getNodes().get(i);
			}
			config.useClusterServers()
				// 集群状态扫描间隔时间，单位是毫秒
				.setScanInterval(2000)
				.addNodeAddress(clusterNodes)
				.setPassword(redisProperties.getPassword());

		}
		// 哨兵模式自动装配
		else if (redisProperties.getSentinel() != null) {
			String[] sentinelNodes = new String[redisProperties.getSentinel().getNodes().size()];
			for (int i = 0; i < redisProperties.getSentinel().getNodes().size(); i++) {
				sentinelNodes[i] = "redis://" + redisProperties.getSentinel().getNodes().get(i);
			}
			config.useSentinelServers()
				.setMasterName(redisProperties.getSentinel().getMaster())
				.setDatabase(redisProperties.getDatabase())
				.addSentinelAddress(sentinelNodes)
				.setPassword(redisProperties.getPassword())
				.setTimeout((int) redisProperties.getTimeout().toMillis())
				.setMasterConnectionPoolSize(redisProperties.getLettuce().getPool().getMaxActive())
				.setSlaveConnectionPoolSize(redisProperties.getLettuce().getPool().getMaxActive())
				.setCheckSentinelsList(false)
			;

		}
		// 单机模式自动装配
		else {
			config.useSingleServer()
				.setAddress("redis://" + redisProperties.getHost() + ":" + redisProperties.getPort())
				.setDatabase(redisProperties.getDatabase())
				.setTimeout((int) redisProperties.getTimeout().toMillis())
				.setConnectionPoolSize(redisProperties.getLettuce().getPool().getMaxActive())
				.setConnectionMinimumIdleSize(redisProperties.getLettuce().getPool().getMinIdle())
				.setPassword(redisProperties.getPassword());
		}
		return Redisson.create(config);
	}

	/**
	 * 装配locker类，并将实例注入到RedissLockUtil中
	 *
	 * @return distributedLocker
	 */
	@Bean
	@ConditionalOnClass(RedissonClient.class)
	DistributedLocker distributedLocker(RedissonClient redissonClient) {
		DistributedLocker locker = new RedissonDistributedLocker();
		locker.setRedissonClient(redissonClient);
		RedissLockUtil.setLocker(locker);
		return locker;
	}
}
