package cn.meteor.common.redis.dao.impl;

import cn.hutool.core.util.ObjectUtil;
import cn.meteor.common.redis.dao.AbstractRedisDao;
import cn.meteor.common.redis.dao.IRedisDao;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.connection.DataType;
import org.springframework.data.redis.connection.RedisServerCommands;
import org.springframework.data.redis.core.HashOperations;
import org.springframework.data.redis.core.ListOperations;
import org.springframework.data.redis.core.SetOperations;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.data.redis.core.ZSetOperations.TypedTuple;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @author ths
 */
@Slf4j
@SuppressWarnings("all")
public class RedisDaoImpl<K, V> extends AbstractRedisDao<K, V> implements IRedisDao<K, V> {

	@Override
	public int getSize() {
		Long size = redisTemplate.execute(RedisServerCommands::dbSize);
		return size == null ? 0 : size.intValue();
	}

	@Override
	public DataType type(final K key) {
		return redisTemplate.type(key);
	}

	@Override
	public boolean existsKey(final K key) {
		Boolean existsKey = redisTemplate.hasKey(key);
		return existsKey == null ? Boolean.FALSE : existsKey;
	}

	@Override
	public void renameKey(K oldKey, K newKey) {
		if (existsKey(oldKey)) {
			redisTemplate.rename(oldKey, newKey);
		}
	}

	@Override
	public Set<K> keys(final K pattern) {
		return redisTemplate.keys(pattern);
	}

	@Override
	public int count(final K key) {
		Set<K> objects = redisTemplate.keys(key);
		return CollectionUtils.isEmpty(objects) ? 0 : objects.size();
	}

	@Override
	public boolean delete(final K key) {
		Boolean result = redisTemplate.delete(key);
		return result == null ? Boolean.FALSE : result;
	}

	@Override
	public void delete(final K[] keys) {
		if (keys == null || keys.length < 1) {
			return;
		}
		for (K key : keys) {
			boolean result = delete(key);
			if (!result && log.isErrorEnabled()) {
				log.error("redis批量删除失败，key = {}", key);
			}
		}
	}

	@Override
	public long deletePattern(final K pattern) {
		Set<K> objects = redisTemplate.keys(pattern);
		if (CollectionUtils.isEmpty(objects)) {
			return 0L;
		}
		return delete(objects);
	}


	@Override
	public long delete(final Set<K> keys) {
		Long deleteNum = redisTemplate.delete(keys);
		return deleteNum == null ? 0L : deleteNum;
	}

	@Override
	public boolean set(final K key, V value) {
		boolean result = false;
		try {
			ValueOperations<K, V> operations = redisTemplate.opsForValue();
			operations.set(key, value);
			result = true;
		} catch (Exception e) {
			log.error("无法执行redis set操作key:{}", key, e);
		}
		return result;
	}

	@Override
	public boolean incr(K key, Long value) {
		boolean result = false;
		try {
			ValueOperations<K, V> operations = redisTemplate.opsForValue();
			operations.increment(key, value);
			result = true;
		} catch (Exception e) {
			log.error("将key中储存的数字加上指定的增量值(操作字符串)key:{}", key, e);
		}
		return result;
	}

	@Override
	public boolean set(final K key, V value, Long expireTime, TimeUnit unit) {
		boolean result = false;
		try {
			ValueOperations<K, V> operations = redisTemplate.opsForValue();
			operations.set(key, value);
			redisTemplate.expire(key, expireTime, unit);
			result = true;
		} catch (Exception e) {
			log.error("无法将写入缓存设置时效时间:key:{}", key);
		}
		return result;
	}

	@Override
	public boolean set(final K key, V value, Long expireTime) {
		return set(key, value, expireTime, TimeUnit.SECONDS);
	}

	@Override
	public boolean updateExpireTime(final K key, Long expireTime, TimeUnit unit) {
		boolean result = false;
		try {
			redisTemplate.expire(key, expireTime, unit);
			result = true;
		} catch (Exception e) {
			log.error("无法更新写入缓存设置时效时间:key:{}", key);
		}
		return result;
	}

	@Override
	public boolean updateExpireTime(final K key, Long expireTime) {
		return updateExpireTime(key, expireTime, TimeUnit.SECONDS);
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T> T get(final K key, Class<T> clazz) {
		Object value = redisTemplate.opsForValue().get(key);
		if (ObjectUtil.isNull(value)) {
			return null;
		}
		return (T) value;
	}

	@Override
	public void hSet(K key, K hashKey, V value) {
		HashOperations<K, K, V> hash = redisTemplate.opsForHash();
		hash.put(key, hashKey, value);
	}

	@Override
	public void hIncr(K key, K hashKey, Long value) {
		HashOperations<K, K, V> hash = redisTemplate.opsForHash();
		hash.increment(key, hashKey, value);
	}

	@Override
	public void hSetAll(K key, Map<K, V> map) {
		HashOperations<K, K, V> hash = redisTemplate.opsForHash();
		hash.putAll(key, map);
	}

	@Override
	public Map<K, V> hGet(K key) {
		HashOperations<K, K, V> hash = redisTemplate.opsForHash();
		return hash.entries(key);
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T> T hGet(K key, K hashKey, Class<T> clazz) {
		HashOperations<K, K, V> hash = redisTemplate.opsForHash();
		V value = hash.get(key, hashKey);
		if (ObjectUtil.isNull(value)) {
			return null;
		}
		return (T) value;
	}

	@Override
	public long hDel(K key, K... hashKey) {
		HashOperations<K, K, V> hash = redisTemplate.opsForHash();
		return hash.delete(key, (Object) hashKey);
	}

	@Override
	public long hSize(K key) {
		HashOperations<K, K, V> hash = redisTemplate.opsForHash();
		return hash.size(key);
	}

	@Override
	@SuppressWarnings("unchecked")
	public List<V> hValues(K key) {
		return (List<V>) redisTemplate.opsForHash().values(key);
	}

	@Override
	public long lSize(K key) {
		ListOperations<K, V> list = redisTemplate.opsForList();
		Long size = list.size(key);
		return size == null ? 0L : size;
	}

	@Override
	public V lIndexFirst(K key) {
		ListOperations<K, V> list = redisTemplate.opsForList();
		return list.index(key, 0);
	}

	@Override
	public V lIndex(K k, long index) {
		ListOperations<K, V> list = redisTemplate.opsForList();
		return list.index(k, index);
	}

	@Override
	public boolean lLeftPush(K key, V value) {
		ListOperations<K, V> list = redisTemplate.opsForList();
		Long result = list.leftPush(key, value);
		return result == null ? Boolean.FALSE : Boolean.TRUE;
	}

	@Override
	public boolean lLeftPush(K key, V value, boolean bool) {
		ListOperations<K, V> list = redisTemplate.opsForList();
		if (bool) {
			Long size = list.size(key);
			list.remove(key, size == null ? 0L : size, value);
		}
		Long result = list.leftPush(key, value);
		return result == null ? Boolean.FALSE : Boolean.TRUE;
	}

	@Override
	public long lLeftPushAll(K key, List<V> lst) {
		ListOperations<K, V> list = redisTemplate.opsForList();
		Long num = list.leftPushAll(key, lst);
		return num == null ? 0L : num;
	}

	@Override
	public boolean lRightPush(K key, V value) {
		ListOperations<K, V> list = redisTemplate.opsForList();
		Long result = list.rightPush(key, value);
		return result == null ? Boolean.FALSE : Boolean.TRUE;
	}

	@Override
	public boolean lRightPush(K key, V value, boolean bool) {
		ListOperations<K, V> list = redisTemplate.opsForList();
		if (bool) {
			Long size = list.size(key);
			list.remove(key, size == null ? 0L : size, value);
		}
		Long result = list.rightPush(key, value);
		return result == null ? Boolean.FALSE : Boolean.TRUE;
	}

	@Override
	public long lRightPushAll(K key, List<V> lst) {
		ListOperations<K, V> list = redisTemplate.opsForList();
		Long num = list.rightPushAll(key, lst);
		return num == null ? 0L : num;
	}

	@Override
	public V lLeftPop(K key) {
		return redisTemplate.opsForList().leftPop(key);
	}

	@Override
	public V lRightPop(K key) {
		return redisTemplate.opsForList().rightPop(key);
	}

	@Override
	public long lRemove(K key, long count) {
		return lRemove(key, count, null);
	}

	@Override
	public long lRemove(K key, long count, V value) {
		if (value == null) {
			return 0L;
		}
		ListOperations<K, V> list = redisTemplate.opsForList();
		Long num = list.remove(key, count, value);
		return num == null ? 0L : num;
	}

	@Override
	public long lRemove(K key, V value) {
		ListOperations<K, V> list = redisTemplate.opsForList();
		Long size = list.size(key);
		Long num = list.remove(key, size == null ? 0L : size, value);
		return num == null ? 0L : num;
	}

	@Override
	public List<V> lRange(K key) {
		ListOperations<K, V> list = redisTemplate.opsForList();
		Long size = list.size(key);
		return list.range(key, 0, size == null ? 0L : size);
	}

	@Override
	public List<V> lRange(K k, long start, long end) {
		return redisTemplate.opsForList().range(k, start, end);
	}

	@Override
	public long sAdd(K key, V... value) {
		SetOperations<K, V> set = redisTemplate.opsForSet();
		Long num = set.add(key, value);
		return num == null ? 0L : num;
	}

	@Override
	public V sRandomMember(K key) {
		SetOperations<K, V> set = redisTemplate.opsForSet();
		return set.randomMember(key);
	}

	@Override
	public List<V> sRandomMembers(K key, long count) {
		SetOperations<K, V> set = redisTemplate.opsForSet();
		return set.randomMembers(key, count);
	}

	@Override
	public Set<V> sDistinctRandomMembers(K key, long count) {
		SetOperations<K, V> set = redisTemplate.opsForSet();
		return set.distinctRandomMembers(key, count);
	}

	@Override
	public V sPop(K key) {
		SetOperations<K, V> set = redisTemplate.opsForSet();
		return set.pop(key);
	}

	@Override
	public long sRemove(K key, V... values) {
		SetOperations<K, V> set = redisTemplate.opsForSet();
		Long num = set.remove(key, values);
		return num == null ? 0L : num;
	}

	@Override
	public Set<V> sMembers(K key) {
		SetOperations<K, V> set = redisTemplate.opsForSet();
		return set.members(key);
	}

	@Override
	public boolean sIsMember(K key, V value) {
		SetOperations<K, V> set = redisTemplate.opsForSet();
		Boolean exists = set.isMember(key, value);
		return exists == null ? Boolean.FALSE : exists;
	}

	@Override
	public long sSize(K key) {
		SetOperations<K, V> set = redisTemplate.opsForSet();
		Long size = set.size(key);
		return size == null ? 0L : size;
	}

	@Override
	public Set<V> sDifference(K key, K otherKey) {
		SetOperations<K, V> set = redisTemplate.opsForSet();
		return set.difference(key, otherKey);
	}

	@Override
	public boolean zAdd(K key, V value, double score) {
		Boolean add = redisTemplate.opsForZSet().add(key, value, score);
		return add == null ? Boolean.FALSE : add;
	}

	@Override
	public long zAdd(K key, Set<TypedTuple<V>> tuples) {
		Long num = redisTemplate.opsForZSet().add(key, tuples);
		return num == null ? 0L : num;
	}

	@Override
	public long zRemove(K key, V... values) {
		Long num = redisTemplate.opsForZSet().remove(key, values);
		return num == null ? 0L : num;
	}

	@Override
	public Set<V> zRangeByScore(K key, double lowScore, double highScore) {
		return redisTemplate.opsForZSet().rangeByScore(key, lowScore, highScore);
	}
}
