package cn.meteor.common.redis.aspectj;

import cn.meteor.common.enums.ResultType;
import cn.meteor.common.model.R;
import cn.meteor.common.redis.annotation.RedisLock;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.data.redis.RedisProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;

import java.lang.reflect.Method;

/**
 * 使用 aop 切面实现redis分布式锁注解
 *
 * @author ths
 */
@Slf4j
@Aspect
@ConditionalOnClass(RedissonClient.class)
@EnableConfigurationProperties(RedisProperties.class)
public class RedisLockAspect extends BaseCacheAspect {

	@Autowired
	private RedissonClient redissonClient;

	@Pointcut("@annotation(redisLock)")
	public void pointCut(RedisLock redisLock) {
	}

	@SneakyThrows
	@Around("pointCut(redisLock)")
	public Object interceptor(ProceedingJoinPoint point, RedisLock redisLock) {
		String key = redisLock.preKey() + ":" + getKey(point, redisLock.key());
		RLock lock = redissonClient.getLock(key);
		boolean isLock = lock.tryLock(redisLock.timeBlock(), redisLock.timeOut(), redisLock.timeUnit());
		if (isLock) {
			try {
				log.info("获取到锁，key = {}", key);
				return point.proceed();
			} finally {
				lock.unlock();
			}
		}
		log.error("没有获取到锁，key = {}", key);
		return R.failure(ResultType.CRUD_LOCK_OPERATE);
	}

	/**
	 * 是否存在注解，如果存在就获取
	 */
	private RedisLock getRedisLock(JoinPoint joinPoint) {
		Signature signature = joinPoint.getSignature();
		MethodSignature methodSignature = (MethodSignature) signature;
		Method method = methodSignature.getMethod();
		if (method != null) {
			return method.getAnnotation(RedisLock.class);
		}
		return null;
	}

}
