package cn.meteor.common.redis.annotation;

import cn.meteor.common.redis.constant.LockKeys;

import java.lang.annotation.*;
import java.util.concurrent.TimeUnit;

/**
 * 基于redis的分布式锁注解（没有看门狗，必须设置超时时间，防止客户端宕机后，死锁发生）
 * <p>
 * 注意：不要在同一个类用普通方法调用加锁的方法，建议拆分调用（AOP内部调用
 * 失效原因： 方法 A 被调用，是基于 AOP 生成的 代理对象 进行的调用；方法 B 调用方法 A ，是 this 目标对象 直接调用，并不是代理对象进行调用）
 * 参考文章：https://blog.csdn.net/u013151053/article/details/106124048
 * <p>
 * Redis 分布式锁固然优秀，然而却并不是无懈可击的。
 * 试想假如有某个操作在 Redis 集群的某节点上创建了锁，然而在集群同步完成前该节点挂掉，那么锁就失效了。
 * 基于此，Redis 的作者给出了“RedLock”方案，大致来讲是通过构造多个 Redis 集群，并多重上锁的方案，来降低故障的概率。
 * Dr. Martin Kleppmann并不认为 Redlock 能解决故障，并写了篇文章来论证，详情不在本文展开，请参考相关资料。
 *
 * @author ths
 */
@Target({ElementType.METHOD})
@Retention(RetentionPolicy.RUNTIME)
@Documented
public @interface RedisLock {

	/**
	 * 默认锁的前缀
	 *
	 * @return 锁前缀
	 */
	String preKey() default LockKeys.PRE_DEFAULT_LOCK;

	/**
	 * 分布式锁的键
	 *
	 * @return 分布式锁的键
	 */
	String key() default LockKeys.DEFAULT_LOCK;

	/**
	 * 默认过期时间，默认600秒
	 *
	 * @return 过期时间
	 */
	long timeOut() default LockKeys.DEFAULT_TIMEOUT;

	/**
	 * 尝试在指定时间内加锁(可以理解为阻塞时间)
	 *
	 * @return 最大尝试时间
	 */
	long timeBlock() default LockKeys.DEFAULT_TIME_BLOCK;

	/**
	 * 时间格式，默认：秒
	 *
	 * @return 时间单位
	 */
	TimeUnit timeUnit() default TimeUnit.SECONDS;
}
