package cn.meteor.common.model;// Copyright (C), Created on 2021-04-03

import cn.hutool.core.util.ObjectUtil;
import cn.meteor.common.enums.ResultType;
import cn.meteor.common.exception.ServiceException;
import cn.meteor.common.interfaces.IResultType;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.Accessors;
import lombok.experimental.SuperBuilder;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;

import java.io.Serializable;
import java.util.Objects;

/**
 * 统一响应对象
 *
 * @author ths
 * @since 1.0.0
 */
@Slf4j
@Schema(title = "响应模型")
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties({"success", "failure", "process"})
@Accessors(chain = true)
@Data
@SuperBuilder
@NoArgsConstructor
@AllArgsConstructor
public class R<T> implements Serializable {

	private static final long serialVersionUID = 1L;

	/**
	 * code：状态码。无论是否成功，必须返回。
	 * <p>
	 * 成功时，状态码为 0
	 * 失败时，对应业务的错误码。
	 */
	@Schema(name = "状态码", required = true)
	private int code;
	/**
	 * 响应数据：成功时，返回该字段
	 */
	@Schema(name = "响应数据")
	@JsonInclude(JsonInclude.Include.NON_EMPTY)
	private T data;
	/**
	 * 响应信息：失败时，返回该字段
	 */
	@Schema(name = "响应信息", required = true)
	private String message;

	public static <T> R<T> success() {
		return success(null);
	}

	public static <T> R<T> success(T data) {
		return create(ResultType.SUCCESS, data);
	}

	public static <T> R<T> failure() {
		return failure(ResultType.FAILED);
	}

	public static <T> R<T> failure(String errorMessage) {
		return failure(errorMessage, null);
	}

	public static <T> R<T> failure(String errorMessage, T data) {
		return create(ResultType.FAILED, data).setMessage(errorMessage);
	}

	public static <T> R<T> failure(IResultType resultEnum) {
		return create(resultEnum);
	}

	public static <T> R<T> failure(ServiceException e) {
		if (ObjectUtil.isNotNull(e.getResultEnum())) {
			return create(e.getResultEnum());
		}
		return failure(e.getMessage());
	}

	public static <T> R<T> failure(Logger log, IResultType resultEnum, Exception e) {
		log.error(resultEnum.getWarning(), e);
		return create(resultEnum);
	}

	public static <T> R<T> failure(Logger log, String errorMessage, Exception e) {
		log.error(errorMessage, e);
		return failure(errorMessage);
	}

	public static <T> R<T> failure(Integer influenceLine) {
		if (ObjectUtil.isNull(influenceLine) || influenceLine < 1) {
			return failure(ResultType.FAILED);
		}
		return success();
	}

	public static <T> void throwError(String errorMessage) {
		throw new ServiceException(errorMessage);
	}

	private static <T> R<T> create(IResultType resultEnum) {
		return create(resultEnum, null);
	}

	private static <T> R<T> create(IResultType resultEnum, T data) {
		return new R<T>()
			.setCode(resultEnum.getCode())
			.setMessage(resultEnum.getMessage())
			.setData(data);
	}

	@JsonIgnore
	public boolean isSuccess() {
		return Objects.equals(ResultType.SUCCESS.getCode(), this.code);
	}

	@JsonIgnore
	public boolean isFailure() {
		return !isSuccess();
	}

}
