package cn.meteor.common.thread;

import lombok.extern.slf4j.Slf4j;

import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 线程池: 已验证，可以生效
 */
@SuppressWarnings("all")
@Slf4j
public class ThreadPool {
	//核心线程数
	public static int corePoolSize = 50;
	//最大线程数 最大线程数一般设为2N+1最好，N是CPU核数 当前生产为8核心数
	public static int maxPoolSize = 150;
	private static ThreadPoolExecutor mExecutor;
	//默认最大队列数量
	private static int maxQueueSize = 2000;
	//空闲线程等待时间（秒）
	private static long keepAliveTime = 1000 * 60;
	//最小等待时间（毫秒）
	private static long minWaitTime = 1000;
	//最大等待次数
	private static long maxWaitCount = 500;

	public static void init(
		int corePoolSize,
		int maximumPoolSize,
		long keepAliveTime) {
		if (corePoolSize > 50) {
			ThreadPool.corePoolSize = corePoolSize;
		}
		if (maximumPoolSize > 150) {
			ThreadPool.maxPoolSize = maximumPoolSize;
		}
		if (ThreadPool.keepAliveTime > keepAliveTime) {
			ThreadPool.keepAliveTime = keepAliveTime;
		}
	}


	public static void execute(Runnable runnable) {
		if (runnable == null) {
			return;
		}
		getThreadPool();
		getThreadPoolIsNext(mExecutor, "ThreadPool");
		mExecutor.execute(runnable);
	}


	/**
	 * 获取线程池
	 *
	 * @return 线程执行类
	 */
	public static ThreadPoolExecutor getThreadPool() {
		if (mExecutor == null) {
			synchronized (ThreadPool.class) {
				if (mExecutor == null) {
					mExecutor = new ThreadPoolExecutor(
						// 核心线程数
						corePoolSize,
						// 最大线程数
						maxPoolSize,
						// 闲置线程存活时间
						keepAliveTime,
						// 时间单位
						TimeUnit.MILLISECONDS,
						// 线程队列
						new LinkedBlockingDeque<Runnable>(maxQueueSize),
						// 线程工厂
						new ClabThreadFactory(),
						// 拒绝策略
						new ThreadPoolExecutor.DiscardPolicy()
					);
				}
			}
		}
		return mExecutor;
	}

	/**
	 * 判断激活的线程数量与最大线程的比列 如果大于80% 则暂停1秒
	 *
	 * @param cachedThreadPool
	 */
	private static void getThreadPoolIsNext(ThreadPoolExecutor cachedThreadPool, String classMethod) {
		boolean isNext = true;
		if (cachedThreadPool == null) {
			return;
		}
		int waitCount = 0;//等待次数
		while (isNext) {
			int maxPoolSize = cachedThreadPool.getMaximumPoolSize();
			int poolSize = cachedThreadPool.getPoolSize();
			if (maxPoolSize <= 0) {
				break;
			}
			if (poolSize <= 0) {
				break;
			}
			double poolNum = poolSize / maxPoolSize;
			if (waitCount > maxWaitCount) {
				getMonitorThreadPoolInfo(cachedThreadPool, waitCount, minWaitTime, classMethod);//发送监控数据
				break;
			}
			if (poolNum > 0.8) {
				waitCount = waitCount + 1;
				try {
					Thread.sleep(minWaitTime);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			} else {
				isNext = false;
			}
		}
		if (waitCount > 0) {
			getMonitorThreadPoolInfo(cachedThreadPool, waitCount, minWaitTime, classMethod);//发送监控数据
		}
	}

	/**
	 * 发送监控数据
	 *
	 * @param cachedThreadPool
	 */
	private static void getMonitorThreadPoolInfo(ThreadPoolExecutor cachedThreadPool, int waitCount, long minWaitTime, String classMethod) {
		if (cachedThreadPool == null) {
			return;
		}
		int activeCount = cachedThreadPool.getActiveCount();
		long completedTaskCount = cachedThreadPool.getCompletedTaskCount();
		int corePoolSize = cachedThreadPool.getCorePoolSize();
		int poolSize = cachedThreadPool.getPoolSize();
		long taskCount = cachedThreadPool.getTaskCount();
		int maximumPoolSize = cachedThreadPool.getMaximumPoolSize();
		StringBuilder stringBuilder = new StringBuilder();
		stringBuilder.append("\n\n【线程池监控】");
		stringBuilder.append("\n\n执行位置：").append(classMethod);
		stringBuilder.append("\n\n 等待睡眠次数=").append(waitCount);
		stringBuilder.append("\n\n 每次睡眠时间(毫秒)=").append(minWaitTime);
		stringBuilder.append("\n\n 配置最大线程数=").append(maximumPoolSize);
		stringBuilder.append("\n\n 当前激活线程数=").append(activeCount);
		stringBuilder.append("\n\n 提交的任务数=").append(taskCount);
		stringBuilder.append("\n\n 已经完成的任务数量=").append(completedTaskCount);
		stringBuilder.append("\n\n 核心线程数=").append(corePoolSize);
		stringBuilder.append("\n\n 当前运行线程数=").append(poolSize);
		log.info("==============================================》" + stringBuilder.toString());
	}

	/**
	 * getExceptionMsg:(根据异常获取关键异常信息字符串)
	 *
	 * @param throwable 异常
	 * @return 返回错误日志
	 */
	public static String getExceptionMsg(Throwable throwable) {
		if (throwable == null) {
			return null;
		}
		StringBuilder stackMessage = new StringBuilder();
		StackTraceElement[] stackTrace = throwable.getStackTrace();
		if (stackTrace == null || stackTrace.length < 1) {
			return null;
		}
		int len = stackTrace.length;
		for (int i = 0; i < len; i++) {
			if (i > 6) {
				break;
			}
			String moduleName = moduleName = stackTrace[i].getClassName();
			String methodName = stackTrace[i].getMethodName();
			Class<?> aClass = null;
			try {
				aClass = Class.forName(moduleName);
			} catch (ClassNotFoundException e) {
			}
			int lineNumber = stackTrace[i].getLineNumber();
			String simpleName = null;
			if (aClass != null) {
				simpleName = "(" + aClass.getSimpleName() + ".java " + lineNumber + ")";
			}
			stackMessage.append(moduleName).append(" ").append(methodName).append(simpleName).append("\n\n");
			if (i == 0) {
				stackMessage.append("错误信息:" + throwable.getMessage()).append("\n\n");
			}
		}
		return stackMessage.toString();
	}

	/**
	 * 线程工厂
	 */
	private static class ClabThreadFactory implements ThreadFactory {
		private AtomicInteger count = new AtomicInteger(0);

		@Override
		public Thread newThread(Runnable r) {
			Thread t = new Thread(r);
			String threadName = "clab_thread_pool_" + ThreadPool.class.getSimpleName() + count.addAndGet(1);
			t.setName(threadName);
			return t;
		}
	}

	/**
	 * 重写拒绝机制，改为阻塞提交。保证不抛弃一个任务
	 */
	private static class ClabRejectedExecutionHandler implements RejectedExecutionHandler {
		@Override
		public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {
			try {
				// 核心改造点，由blockingqueue的offer改成put阻塞方法
				executor.getQueue().put(r);
			} catch (InterruptedException e) {
				log.error(getExceptionMsg(e));
			}
		}
	}

}
