/**
 * Copyright 2010 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package zkclient;

import org.apache.zookeeper.ZooKeeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import zkclient.exception.ZkInterruptedException;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * All listeners registered at the {@link ZookeeperClient} will be notified from this event thread. This is to prevent
 * dead-lock situations. The {@link ZookeeperClient} pulls some information out of the {@link ZooKeeper} events to signal
 * {@link ZookeeperLock} conditions. Re-using the {@link ZooKeeper} event thread to also notify {@link ZookeeperClient} listeners,
 * would stop the ZkClient from receiving events from {@link ZooKeeper} as soon as one of the listeners blocks (because
 * it is waiting for something). {@link ZookeeperClient} would then for instance not be able to maintain it's connection state
 * anymore.
 */
class ZookeeperEventThread extends Thread {

    private static final Logger LOG = LoggerFactory.getLogger(ZookeeperEventThread.class);

    private BlockingQueue<ZkEvent> events = new LinkedBlockingQueue<ZkEvent>();

    private static AtomicInteger EVENTID = new AtomicInteger(0);

    static abstract class ZkEvent {

        private String description;

        public ZkEvent(String description) {
            this.description = description;
        }

        public abstract void run() throws Exception;

        @Override
        public String toString() {
            return "ZkEvent[" + description + "]";
        }
    }

    ZookeeperEventThread(String name) {
        setDaemon(true);
        setName("ZkClient-EventThread-" + getId() + "-" + name);
    }

    @Override
    public void run() {
        LOG.info("Starting ZkClient event thread.");
        try {
            while (!isInterrupted()) {
                ZkEvent zkEvent = events.take();
                int eventId = EVENTID.incrementAndGet();
                LOG.debug("Delivering event #" + eventId + " " + zkEvent);
                try {
                    zkEvent.run();
                } catch (InterruptedException e) {
                    interrupt();
                } catch (ZkInterruptedException e) {
                    interrupt();
                } catch (Throwable e) {
                    LOG.error("Error handling event " + zkEvent, e);
                }
                LOG.debug("Delivering event #" + eventId + " done");
            }
        } catch (InterruptedException e) {
            LOG.info("Terminate ZkClient event thread.");
        }
    }

    public void send(ZkEvent event) {
        if (!isInterrupted()) {
            LOG.debug("New event: " + event);
            events.add(event);
        }
    }
}
