/**
 * Copyright 2010 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package zkclient;

import org.apache.zookeeper.*;
import org.apache.zookeeper.ZooDefs.Ids;
import org.apache.zookeeper.ZooKeeper.States;
import org.apache.zookeeper.client.ConnectStringParser;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import zkclient.exception.ZkException;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.util.AbstractMap.SimpleEntry;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

public class ZookeeperConnection implements IZookeeperConnection {

    private static final Logger LOG = LoggerFactory.getLogger(ZookeeperConnection.class);

    /** It is recommended to use quite large sessions timeouts for ZooKeeper. */
    private static final int DEFAULT_SESSION_TIMEOUT = 50000;

    private ZooKeeper zk = null;
    
    private Lock zookeeperLock = new ReentrantLock();

    private final String servers;
    
    private String resolvedServers;
    
    private final int sessionTimeOut;

    public ZookeeperConnection(String zkServers) {
        this(zkServers, DEFAULT_SESSION_TIMEOUT);
    }

    public ZookeeperConnection(String zkServers, int sessionTimeOut) {
        this.servers = zkServers;
        this.sessionTimeOut = sessionTimeOut;
    }

    @Override
    public void connect(Watcher watcher) {
        zookeeperLock.lock();
        try {
            if (zk != null) {
                throw new IllegalStateException("zk client has already been started");
            }
            try {
                LOG.debug("Resolving hosts from configured hosts " + servers);
                resolvedServers = resolveHosts();
                LOG.debug("Creating new ZookKeeper instance to connect to resolved hosts " + resolvedServers + ".");
                System.setProperty("zookeeper.sasl.client", "false");//true开启sasl认证，false关闭认证
                zk = new ZooKeeper(resolvedServers, sessionTimeOut, watcher);
            } catch (IOException e) {
                throw new ZkException("Unable to connect to " + resolvedServers, e);
            }
        } finally {
            zookeeperLock.unlock();
        }
    }

    @Override
    public void close() throws InterruptedException {
        zookeeperLock.lock();
        try {
            if (zk != null) {
                LOG.debug("Closing ZooKeeper connected to " + servers);
                zk.close();
                zk = null;
            }
        } finally {
            zookeeperLock.unlock();
        }
    }

    @Override
    public String create(String path, byte[] data, CreateMode mode) throws KeeperException, InterruptedException {
        return zk.create(path, data, Ids.OPEN_ACL_UNSAFE, mode);
    }

    @Override
    public String create(String path, byte[] data, List<ACL> acl, CreateMode mode) throws KeeperException, InterruptedException {
        return zk.create(path, data, acl, mode);
    }

    @Override
    public void delete(String path) throws InterruptedException, KeeperException {
        zk.delete(path, -1);
    }

    @Override
    public void delete(String path, int version) throws InterruptedException, KeeperException {
        zk.delete(path, version);
    }

    @Override
    public boolean exists(String path, boolean watch) throws KeeperException, InterruptedException {
        return zk.exists(path, watch) != null;
    }

    @Override
    public List<String> getChildren(final String path, final boolean watch) throws KeeperException, InterruptedException {
        return zk.getChildren(path, watch);
    }

    @Override
    public byte[] readData(String path, Stat stat, boolean watch) throws KeeperException, InterruptedException {
        return zk.getData(path, watch, stat);
    }

    public void writeData(String path, byte[] data) throws KeeperException, InterruptedException {
        writeData(path, data, -1);
    }

    @Override
    public void writeData(String path, byte[] data, int version) throws KeeperException, InterruptedException {
        zk.setData(path, data, version);
    }

    @Override
    public Stat writeDataReturnStat(String path, byte[] data, int version) throws KeeperException, InterruptedException {
        return zk.setData(path, data, version);
    }

    @Override
    public States getZookeeperState() {
        return zk != null ? zk.getState() : null;
    }

    public ZooKeeper getZookeeper() {
        return zk;
    }

    @Override
    public long getCreateTime(String path) throws KeeperException, InterruptedException {
        Stat stat = zk.exists(path, false);
        if (stat != null) {
            return stat.getCtime();
        }
        return -1;
    }

    @Override
    public String getServers() {
        return servers;
    }

    @Override
    public String getResolvedServers() {
        return resolvedServers;
    }
    
    @Override
    public List<OpResult> multi(Iterable<Op> ops) throws KeeperException, InterruptedException {
        return zk.multi(ops);
    }

    @Override
    public void addAuthInfo(String scheme, byte[] auth) {
        zk.addAuthInfo(scheme, auth);
    }

    @Override
    public void setAcl(String path, List<ACL> acl, int version) throws KeeperException, InterruptedException {
        zk.setACL(path, acl, version);
    }

    @SuppressWarnings("rawtypes")
	@Override
    public Map.Entry<List<ACL>, Stat> getAcl(String path) throws KeeperException, InterruptedException {
        Stat stat = new Stat();
        List<ACL> acl = zk.getACL(path, stat);
        return new SimpleEntry(acl, stat);
    }

    private String resolveHosts() {
        ConnectStringParser connectStringParser = new ConnectStringParser(servers);
        Collection<InetSocketAddress> serverAddresses = connectStringParser.getServerAddresses();
        List<String> resolvedHosts = new ArrayList<>();
        for (InetSocketAddress address : serverAddresses) {
            try {
                InetAddress ia = address.getAddress();
                String addr = (ia != null) ? ia.getHostAddress() : address.getHostString();
                //this will throw an exception if the host is not resolved. Catch exception.
                InetAddress.getAllByName(addr);
                resolvedHosts.add(address.getHostString() + ":" + address.getPort());
            } catch (UnknownHostException ex) {
                LOG.warn("No IP address found for server: {}", address, ex);
            }
        }
        String resolvedConnectionString = String.join(",", resolvedHosts);
        if (connectStringParser.getChrootPath() != null) {
            resolvedConnectionString += connectStringParser.getChrootPath();
        }
        return resolvedConnectionString;
    }
}
